<?php

namespace Drupal\draggableviews\Handler;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\draggableviews\DraggableViewsHandlerInterface;
use Drupal\views\ViewExecutable;

/**
 * Native handler for draggableviews.
 *
 * This handler saves weights to a custom database table.
 */
class NativeHandler implements DraggableViewsHandlerInterface {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new NativeHandler.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(Connection $database, MessengerInterface $messenger) {
    $this->database = $database;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(ViewExecutable $view, $index) {
    $row = $view->result[$index];
    return $row->draggableviews_structure_weight ?? 0;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array &$form, FormStateInterface $form_state, ViewExecutable $view) {
    $input = $form_state->getUserInput();

    if (!isset($input['draggableviews']) || !is_array($input['draggableviews'])) {
      return;
    }

    $view_name = $view->id();
    $view_display = $view->current_display;

    // Get view arguments.
    $arguments = $view->args;
    if ($exposed_input = $view->getExposedInput()) {
      $arguments = array_merge($arguments, $exposed_input);
      ksort($arguments);
    }

    // Remove items_per_page argument.
    unset($arguments['items_per_page']);

    $args_string = json_encode($arguments);

    // Sort items by weight before saving.
    uasort($input['draggableviews'], function ($a, $b) {
      return ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0);
    });

    // Save records to our custom table.
    $weight = 0;

    foreach ($input['draggableviews'] as $item) {
      // Make sure id is available.
      if (!isset($item['id'])) {
        continue;
      }

      try {
        // Delete previous order record.
        $this->database->delete('draggableviews_structure')
          ->condition('view_name', $view_name)
          ->condition('view_display', $view_display)
          ->condition('args', $args_string)
          ->condition('entity_id', $item['id'])
          ->execute();

        // Create new order record.
        $record = [
          'view_name' => $view_name,
          'view_display' => $view_display,
          'args' => $args_string,
          'entity_id' => $item['id'],
          'weight' => $weight,
        ];

        // If parent element exists, save it.
        if (isset($item['parent'])) {
          $record['parent'] = $item['parent'];
        }

        $this->database->insert('draggableviews_structure')
          ->fields($record)
          ->execute();

        $weight++;
      }
      catch (\Exception $e) {
        $this->messenger->addError(t('Failed to save order for entity @id: @message', [
          '@id' => $item['id'],
          '@message' => $e->getMessage(),
        ]));
      }
    }

    $this->messenger->addStatus(t('Order saved successfully.'));
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(array $form, FormStateInterface $form_state, array $options) {
    $form['native_info'] = [
      '#markup' => '<div class="messages messages--info">' . t('Native handler stores the order in a custom database table. No additional configuration is needed.') . '</div>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultOptions() {
    return [];
  }

}

