<?php

namespace Drupal\draggableviews\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\draggableviews\DraggableViews;
use Drupal\views\Plugin\views\field\BulkForm;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Defines a draggableviews form element.
 *
 * @ViewsField("draggable_views_field")
 */
class DraggableViewsField extends BulkForm {

  /**
   * The entity manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityManager;

  /**
   * The action storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $actionStorage;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a new DraggableViewsField object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, $entity_type_manager, $language_manager, $messenger, $entity_repository, $route_match, AccountProxyInterface $current_user) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entity_type_manager, $language_manager, $messenger, $entity_repository, $route_match);
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('messenger'),
      $container->get('entity.repository'),
      $container->get('current_route_match'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['draggable_views_hierarchy'] = ['default' => 0];
    $options['draggable_views_handler'] = ['default' => 'native'];
    $options['draggable_views_weight_field'] = ['default' => ''];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    $form['draggable_views_help'] = [
      '#markup' => $this->t("A draggable element will be added to the first table column. You do not have to set this field as the first column in your View.<br>Current handler value: <strong>@handler</strong>",
        ['@handler' => $this->options['draggable_views_handler'] ?? 'native']
      ),
    ];

    $form['draggable_views_hierarchy'] = [
      '#title' => $this->t('Enable hierarchy'),
      '#type' => 'checkbox',
      '#default_value' => $this->options['draggable_views_hierarchy'],
      '#weight' => -1,
    ];

    // Determine current handler value.
    $triggering_element = $form_state->getTriggeringElement();
    $is_handler_change = $triggering_element &&
      isset($triggering_element['#name']) &&
      $triggering_element['#name'] === 'options[draggableviews_views_handler_wrapper][draggable_views_handler]';

    if ($is_handler_change) {
      // During AJAX request triggered by handler change.
      $current_handler = $form_state->getValue(['options', 'draggableviews_views_handler_wrapper', 'draggable_views_handler']);
    }
    else {
      // Initial page load or other triggers - use saved options.
      $current_handler = $this->options['draggable_views_handler'] ?? 'native';
    }

    // Container for handler selection with AJAX wrapper.
    $form['draggableviews_views_handler_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'draggableviews-views-handler-wrapper'],
    ];

    $form['draggableviews_views_handler_wrapper']['draggable_views_handler'] = [
      '#type' => 'select',
      '#title' => $this->t('Handler'),
      '#options' => [
        'native' => $this->t('Native (internal table)'),
        'fieldapi' => $this->t('Field API (entity field)'),
      ],
      '#default_value' => $current_handler,
      '#ajax' => [
        'callback' => [static::class, 'ajaxOptionsRefresh'],
        'wrapper' => 'draggableviews-views-handler-wrapper',
      ],
    ];

    // Add description text.
    $form['draggableviews_views_handler_wrapper']['handler_description'] = [
      '#type' => 'item',
      '#markup' => $this->t('Choose how to save the order.'),
    ];

    // Show weight field selector only when Field API handler is selected.
    if ($current_handler === 'fieldapi') {
      try {
        $handler_factory = \Drupal::service('draggableviews.handler_factory');
        $handler = $handler_factory->getHandler('fieldapi');

        // Set the view executable in form state for the handler to use.
        if ($this->view) {
          $form_state->set('view_executable', $this->view);
        }

        $handler_options = $handler->buildOptionsForm([], $form_state, $this->options);

        foreach ($handler_options as $key => $element) {
          $form['draggableviews_views_handler_wrapper'][$key] = $element;
        }
      }
      catch (\Exception $e) {
        \Drupal::logger('draggableviews')->error('Failed to build handler options: @message', ['@message' => $e->getMessage()]);
        $form['draggableviews_views_handler_wrapper']['error'] = [
          '#markup' => '<div class="messages messages--error">' . $this->t('Error loading field options: @message', ['@message' => $e->getMessage()]) . '</div>',
        ];
      }
    }

    // Don't call parent::buildOptionsForm to avoid adding bulk form fields.
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    // Override parent query() to prevent BulkForm from adding its queries.
    // Draggableviews doesn't need any special query modifications.
  }

  /**
   * {@inheritdoc}
   */
  public function clickSortable() {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  // @codingStandardsIgnoreStart
  public function render_item($count, $item) {
    // @codingStandardsIgnoreEnd
    // Using internal method. @todo Recheck after drupal stable release.
    return Markup::create('<!--form-item-' . $this->options['id'] . '--' . $this->view->row_index . '-->');
  }

  /**
   * {@inheritdoc}
   */
  public function viewsForm(&$form, FormStateInterface $form_state) {
    // Disable form caching.
    $form['#cache']['max-age'] = 0;

    $form[$this->options['id']] = ['#tree' => TRUE];

    $draggableviews = new DraggableViews($this->view);
    $handler = $this->options['draggable_views_handler'] ?? 'native';
    $weight_field = $this->options['draggable_views_weight_field'] ?? '';

    foreach ($this->view->result as $row_index => $row) {
      $entity = $this->getEntity($row);
      if (empty($entity) || !is_object($entity)) {
        continue;
      }

      $form[$this->options['id']][$row_index] = ['#tree' => TRUE];

      // Determine weight based on handler type.
      $weight = $row_index;

      if ($handler === 'fieldapi' && !empty($weight_field) && $entity->hasField($weight_field)) {
        $field_value = $entity->get($weight_field)->value;
        $weight = $field_value ?? $row_index;
      }
      elseif ($handler === 'native') {
        $weight = property_exists($row, 'draggableviews_structure_weight') ? $row->draggableviews_structure_weight : $row_index;
      }

      $range = count($this->view->result);
      $form[$this->options['id']][$row_index]['weight'] = [
        '#type' => 'select',
        '#title' => $this->t('Weight'),
        '#title_display' => 'invisible',
        '#options' => range(-$range, $range),
        '#default_value' => $weight,
        '#attributes' => ['class' => ['draggableviews-weight']],
      ];

      $form[$this->options['id']][$row_index]['id'] = [
        '#type' => 'hidden',
        '#value' => $entity->id(),
        '#attributes' => ['class' => ['draggableviews-id']],
      ];

      $form[$this->options['id']][$row_index]['parent'] = [
        '#type' => 'hidden',
        '#default_value' => $draggableviews->getParent($row_index),
        '#attributes' => ['class' => ['draggableviews-parent']],
      ];
    }

    // Add tabledrag if user has permission.
    if ($this->currentUser && $this->currentUser->hasPermission('access draggableviews')) {
      $fieldGrouping = $draggableviews->fieldGrouping();
      foreach ($fieldGrouping as $key => $row) {
        $options = [
          'table_id' => $draggableviews->getHtmlId($key),
          'action' => 'match',
          'relationship' => $this->options['draggable_views_hierarchy'] === 1 ? 'parent' : 'sibling',
          'group' => 'draggableviews-parent',
          'subgroup' => 'draggableviews-parent',
          'source' => 'draggableviews-id',
        ];
        drupal_attach_tabledrag($form, $options);
      }
    }

    // Always add a submit button for debugging.
    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save order'),
      '#button_type' => 'primary',
    ];
  }

  /**
   * Sets the current_user service.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Current user.
   *
   * @return $this
   */
  public function setCurrentUser(AccountProxyInterface $current_user) {
    $this->currentUser = $current_user;
    return $this;
  }

  /**
   * AJAX callback for handler options refresh.
   */
  public static function ajaxOptionsRefresh(array &$form, FormStateInterface $form_state) {
    return $form['options']['draggableviews_views_handler_wrapper'];
  }


  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(&$form, FormStateInterface $form_state) {
    // Don't call parent to avoid BulkForm validation that expects selected_actions.
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state) {
    $options = $form_state->getValue('options');

    if (isset($options['draggableviews_views_handler_wrapper']['draggable_views_handler'])) {
      $this->options['draggable_views_handler'] = $options['draggableviews_views_handler_wrapper']['draggable_views_handler'];
    }

    if (isset($options['draggableviews_views_handler_wrapper']['draggable_views_weight_field'])) {
      $this->options['draggable_views_weight_field'] = $options['draggableviews_views_handler_wrapper']['draggable_views_weight_field'];
    }

    if (isset($options['draggable_views_hierarchy'])) {
      $this->options['draggable_views_hierarchy'] = $options['draggable_views_hierarchy'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function viewsFormSubmit(&$form, FormStateInterface $form_state) {
    $handler = $this->options['draggable_views_handler'] ?? 'native';

    if ($handler === 'fieldapi') {
      $this->submitFieldApiHandler($form, $form_state);
    }
    else {
      $this->submitNativeHandler($form, $form_state);
    }
  }

  /**
   * Submit handler for Field API handler.
   */
  protected function submitFieldApiHandler(&$form, FormStateInterface $form_state) {
    // Instantiate the FieldApiHandler and call its save method.
    $handler = new \Drupal\draggableviews\Handler\FieldApiHandler(
      \Drupal::entityTypeManager(),
      \Drupal::messenger()
    );

    // Pass the field configuration to the handler.
    $handler->saveWithConfig(
      $form,
      $form_state,
      $this->view,
      $this->options['id'],
      $this->options['draggable_views_weight_field'] ?? ''
    );
  }

  /**
   * Submit handler for Native handler.
   */
  protected function submitNativeHandler(&$form, FormStateInterface $form_state) {
    // Native handler logic - this would save to draggableviews_structure table.
    // TODO: Implement native handler save logic.
  }


}
