<?php

declare(strict_types=1);

namespace Drupal\drext\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Returns responses for Drext routes.
 */
final class DrextController extends ControllerBase {

  /**
   * The controller constructor.
   */
  public function __construct(
    private readonly ModuleExtensionList $extensionListModule,
    private readonly RequestStack $requestStack,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('extension.list.module'),
      $container->get('request_stack'),
    );
  }

  /**
   * Builds the response.
   */
  public function __invoke(): array {
    // Environment variables.
    $request = $this->requestStack->getCurrentRequest();

    $decoupled_kit_info = $this->extensionListModule->getExtensionInfo('decoupled_kit');
    $decoupled_kit_version = (int) explode('.', $decoupled_kit_info['version'])[0];

    $variables = [
      'required_variables' => '# Required',
      'NEXT_PUBLIC_DRUPAL_BASE_URL' => $request->getSchemeAndHttpHost(),
      'NEXT_IMAGE_DOMAIN' => $request->getHost(),
      'DECOUPLED_KIT_API_VERSION' => $decoupled_kit_version,
    ];

    $build['container'] = [
      '#type' => 'container',
    ];

    $build['container']['title'] = [
      '#markup' => '<h3>' . $this->t('Environment variables') . '</h3>',
    ];

    $build['container']['description'] = [
      '#type' => 'html_tag',
      '#tag' => 'p',
      '#value' => $this->t('Copy and paste these values in your <code>.env</code> or <code>.env.local</code> files.'),
    ];

    foreach ($variables as $name => $value) {
      $build['container'][$name] = [
        '#type' => 'inline_template',
        '#template' => "{% if value starts with '#' %}{{ value }} {% else %} {{ name }}={{ value }}{% endif %}<br/>",
        '#context' => [
          'name' => $name,
          'value' => $value,
        ],
      ];
    }

    // Next.js sites list.
    $sites = $this->entityTypeManager()->getStorage('next_site')->loadMultiple();
    if (!empty($sites)) {
      $links = [];
      foreach ($sites as $site) {
        $url = $site->getBaseUrl();
        $links[] = Link::fromTextAndUrl(
          $url,
          Url::fromUri($url, [
            'attributes' => [
              'target' => '_blank',
              'rel' => 'noopener noreferrer',
            ],
          ]))
          ->toRenderable();
      }

      $build['sites'] = [
        '#theme' => 'item_list',
        '#title' => $this->t('Next.js sites'),
        '#items' => $links,
      ];
    }

    // Related modules.
    $build['modules'] = [
      '#type' => 'container',
    ];

    $build['modules']['title'] = [
      '#markup' => '<h3>' . $this->t('Related modules') . '</h3>',
    ];

    $modules_data = [];
    $modules_data['entity.next_site.collection'] = $this->t('Next.js');
    $modules_data['decoupled_kit.dashboard'] = $this->t('Decoupled Kit');

    $modules = [];
    foreach ($modules_data as $route => $name) {
      $modules[] = Link::fromTextAndUrl(
        $name,
        Url::fromRoute($route, [], [
          'attributes' => [
            'target' => '_blank',
            'rel' => 'noopener noreferrer',
          ],
        ]))
        ->toRenderable();
    }

    $build['modules']['list'] = [
      '#theme' => 'item_list',
      '#items' => $modules,
    ];

    return $build;
  }

}
