<?php

namespace Drupal\dripyard_simple_login\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\user\UserStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Flood\FloodInterface;

/**
 * Provides a magic link login form that replaces the default login route.
 *
 * This form repurposes Drupal core's password reset functionality to send
 * one-time login links instead of requiring password authentication.
 */
class MagicLinkLoginForm extends FormBase {

  /**
   * The user storage.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected $userStorage;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The flood service.
   *
   * @var \Drupal\Core\Flood\FloodInterface
   */
  protected $flood;

  /**
   * Constructs a new MagicLinkLoginForm.
   *
   * @param \Drupal\user\UserStorageInterface $user_storage
   *   The user storage.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Flood\FloodInterface $flood
   *   The flood service.
   */
  public function __construct(UserStorageInterface $user_storage, LanguageManagerInterface $language_manager, FloodInterface $flood) {
    $this->userStorage = $user_storage;
    $this->languageManager = $language_manager;
    $this->flood = $flood;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')->getStorage('user'),
      $container->get('language_manager'),
      $container->get('flood')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dripyard_simple_login_magic_link_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#cache']['tags'] = ['rendered'];

    $form['description'] = [
      '#prefix' => '<p>',
      '#markup' => $this->t('Enter your email address. You will receive an email with a link to log in.'),
      '#suffix' => '</p>',
    ];

    $form['name'] = [
      '#type' => 'email',
      '#title' => $this->t('Email address'),
      '#size' => 60,
      '#maxlength' => 254,
      '#required' => TRUE,
      '#attributes' => [
        'autocorrect' => 'off',
        'autocapitalize' => 'off',
        'spellcheck' => 'false',
        'autocomplete' => 'email',
      ],
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Send login link'),
      '#attributes' => ['class' => ['button--primary']],
    ];
    $form['actions']['password'] = [
      '#type' => 'submit',
      '#value' => $this->t('Use password'),
      '#submit' => ['::passwordLoginRedirect'],
      '#limit_validation_errors' => [],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Repurpose core's password reset flood protection for login attempts.
    $identifier = $this->getRequest()->getClientIp();
    $flood_config = $this->config('user.flood');
    if (!$this->flood->isAllowed('user.password_reset_ip', $flood_config->get('ip_limit'), $flood_config->get('ip_window'), $identifier)) {
      $form_state->setErrorByName('name', $this->t('You have issued too many login attempts. Please contact us or try again later.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $name = trim($form_state->getValue('name'));

    $users = $this->userStorage->loadByProperties(['mail' => $name]);
    if (empty($users)) {
      $users = $this->userStorage->loadByProperties(['name' => $name]);
    }

    /** @var \Drupal\user\UserInterface|null $account */
    $account = reset($users);

    // Register flood event using core's password reset flood tracking.
    $identifier = $this->getRequest()->getClientIp();
    $this->flood->register('user.password_reset_ip', $this->config('user.flood')->get('ip_window'), $identifier);

    if ($account && $account->id() && $account->isActive()) {
      // Repurpose core's password_reset mail to send magic login link.
      $mail = _user_mail_notify('password_reset', $account);
      if (!empty($mail)) {
        $this->logger('user')->info('Password reset instructions mailed to %name at %email.', ['%name' => $account->getAccountName(), '%email' => $account->getEmail()]);
      }
    }

    // Add a small random delay to prevent timing attacks for user enumeration.
    usleep(\random_int(10000, 50000));

    // Always show the same message to prevent user enumeration.
    $this->messenger()->addMessage($this->t('Check %email on this device for your login link.', ['%email' => $name]));
    $form_state->setRedirect('<front>');
  }

  /**
   * Submit handler for password login redirect.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function passwordLoginRedirect(array &$form, FormStateInterface $form_state) {
    $form_state->setRedirect('dripyard_simple_login.login_password');
  }

}
