<?php

namespace Drupal\dropbox_sign\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\encryption\EncryptionService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Dropbox Sign settings for this site.
 */
class DropboxSignSettingsForm extends ConfigFormBase {

  use StringTranslationTrait;

  /**
   * The encryption service.
   *
   * @var \Drupal\encryption\EncryptionService
   */
  protected EncryptionService $encryption;

  /**
   * Constructs a DropboxSignSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\encryption\EncryptionService $encryption
   *   The encryption service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    EncryptionService $encryption,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->encryption = $encryption;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('encryption')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'dropbox_sign_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['dropbox_sign.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('dropbox_sign.settings');

    // Encrypted fields.
    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Dropbox Sign API Key'),
      '#default_value' => $config->get('api_key') ? $this->encryption->decrypt($config->get('api_key'), TRUE) : '',
      '#required' => TRUE,
    ];

    $form['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Dropbox Sign Client ID'),
      '#default_value' => $config->get('client_id') ? $this->encryption->decrypt($config->get('client_id'), TRUE) : '',
    ];

    // Non-sensitive fields can use #config_target.
    $form['cc_emails'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CC Email Addresses'),
      '#description' => $this->t('Emails copied on all requests but not signers. Separate multiple addresses with a comma.'),
      '#config_target' => 'dropbox_sign.settings:cc_emails',
    ];

    $form['test_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Test Mode'),
      '#config_target' => 'dropbox_sign.settings:test_mode',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $api_key = $form_state->getValue('api_key');
    if ($api_key) {
      $error_message = dropbox_sign_validate_dropboxsign_api_key($api_key);
      if ($error_message) {
        $form_state->setErrorByName('api_key', $this->t('@error_message', ['@error_message' => $error_message]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Encrypt sensitive fields.
    $encrypted_api_key = $this->encryption->encrypt($form_state->getValue('api_key'), TRUE);
    $encrypted_client_id = $this->encryption->encrypt($form_state->getValue('client_id'), TRUE);

    if (is_null($encrypted_api_key) || is_null($encrypted_client_id)) {
      $this->messenger()->addError($this->t(
        'Failed to encrypt the API Key and/or Client ID. Ensure the Encryption module is enabled and a key is set.'
      ));
      return;
    }

    // Save encrypted fields manually,
    // #config_target fields are saved automatically.
    $this->config('dropbox_sign.settings')
      ->set('api_key', $encrypted_api_key)
      ->set('client_id', $encrypted_client_id)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
