<?php

namespace Drupal\dropwatch\Service;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\dropwatch\Client\DropWatchApiClient;
use Drupal\system\SystemManager;
use Drupal\update\ProjectRelease;
use Drupal\update\UpdateManagerInterface;

class DropWatchService {

  use LoggerChannelTrait;
  use StringTranslationTrait;

  /**
   * Construct a DropWatchService service.
   */
  public function __construct(
    protected DropWatchApiClient $dropWatchApiClient,
    protected ConfigFactory $configFactory,
    protected SystemManager $systemManager,
    protected ModuleExtensionList $moduleExtensionList,
    protected ModuleHandlerInterface $moduleHandler,
    protected Connection $connection,
  ) {}

  /**
   * Build the payload for a site update API request.
   */
  public function buildApiPayload(): array {
    $config = $this->configFactory->get('dropwatch.settings');
    $system_info = $this->getSystemInfo();
    $modules_info = $this->getContribModuleInfo();
    $updates_info = $this->getUpdateInfo();

    $payload['site_url'] = $config->get('site_url');

    // Core details.
    if ($config->get('core')) {
      $payload['core'] = [
        'version' => $system_info['drupal']['value'],
      ];

      if (array_key_exists('drupal', $updates_info)) {
        $payload['core']['recommended_version'] = $updates_info['drupal']['recommended_version'];
        $payload['core']['recommended_version_url'] = $updates_info['drupal']['recommended_version_url'];
        $payload['core']['releases'] = $updates_info['drupal']['releases'];
      }
    }

    // PHP details.
    if ($config->get('php')) {
      $payload['php'] = (object) [
        'version' => $system_info['php']['value'],
        'apcu_available' => $system_info['php_apcu_available']['value'],
        'apcu_enabled' => $system_info['php_apcu_enabled']['value'],
        'memory_limit' => $system_info['php_memory_limit']['value'],
        'opcache' => $system_info['php_opcache']['value'],
      ];
    }

    // Webserver details.
    if ($config->get('web_server')) {
      $payload['web_server'] = (object) [
        'version' => $system_info['webserver']['value'],
      ];
    }

    // Database details.
    if ($config->get('database')) {
      $payload['database'] = (object) [
        'db_system' => $system_info['database_system']['value'],
        'db_version' => $system_info['database_system_version']['value'],
        'db_updates' => $system_info['update']['value'],
      ];
    }

    // Contrib modules details.
    if ($config->get('contrib_modules')) {
      foreach ($modules_info as $machine_name => $module) {
        if ($machine_name === $module['project']) {
          $payload['modules'][$machine_name] = [
            'name' => $module['name'],
            'machine_name' => $machine_name,
            'core_version_requirement' => $module['core_version_requirement'],
            'version' => $module['version'],
            'lifecycle' => $module['lifecycle'],
          ];
        }

        // Add any updates to the module information.
        if (array_key_exists($machine_name, $updates_info) && $module['version'] != $updates_info[$machine_name]['recommended_version']) {
          $payload['modules'][$machine_name]['recommended_version'] = $updates_info[$machine_name]['recommended_version'];
          $payload['modules'][$machine_name]['recommended_version_url'] = $updates_info[$machine_name]['recommended_version_url'];
          $payload['modules'][$machine_name]['releases'] = $updates_info[$machine_name]['releases'];
        }
      }
    }

    // PHP logs.
    if ($config->get('php_logs')) {
      $payload['php_logs'] = $this->getPhpLogs();
    }

    return $payload;
  }

  /**
   * Get system information.
   *
   * @return array
   */
  public function getSystemInfo(): array {
    return $this->systemManager->listRequirements();
  }

  /**
   * Get module information.
   *
   * @return array
   */
  public function getContribModuleInfo(): array {
    $contrib_modules = [];
    $modules = $this->moduleHandler->getModuleList();

    foreach ($modules as $machine_name => $module) {
      $details = $this->moduleExtensionList->getExtensionInfo($machine_name);
      if (!empty($details['project'])) {
        $contrib_modules[$machine_name] = $details;
      }
    }

    // Sort by key alphabetically.
    ksort($contrib_modules, SORT_STRING);

    return $contrib_modules;
  }

  /**
   * Get any update information.
   */
  public function getUpdateInfo(): array {
    $updates_info = [];
    $available = update_get_available(TRUE);

    if (!empty($available)) {
      $project_data = update_calculate_project_data($available);

      foreach ($project_data as $name => $project) {
        // Filter out projects which are up to date already.
        if ($project['status'] == UpdateManagerInterface::CURRENT) {
          continue;
        }

        // If we don't know what to recommend they upgrade to, we should skip
        // the project entirely.
        if (empty($project['recommended'])) {
          continue;
        }

        $recommended_release = ProjectRelease::createFromArray($project['releases'][$project['recommended']]);
        $updates_info[$name] = [
          'recommended_version' => $recommended_release->getVersion(),
          'recommended_version_url' => $recommended_release->getReleaseUrl(),
          'releases' => $project['releases'],
        ];
      }
    }


    return $updates_info;
  }

  /**
   * Get any PHP logs.
   */
  public function getPhpLogs(): array {
    $php_logs = [];
    $query = $this->connection->select('watchdog', 'wd')
      ->fields('wd', [
        'type',
        'message',
        'variables',
        'severity',
      ])
      ->condition('type', 'php');
    $results = $query->execute()->fetchAll();

    foreach ($results as $result) {
      $variables = @unserialize($result->variables);
      $message_details = $this->getMessageDetails($result);

      $php_logs[] = [
        'type' => $variables['%type'],
        'message' => $message_details['message'],
        'backtrace' => $message_details['backtrace'],
        'severity' => $result->severity,
      ];
    }

    return $php_logs;
  }

  /**
   * Send a site update API request to DropWatch.
   */
  public function sendApiRequest(): void {
    $payload = $this->buildApiPayload();

    try {
      $this->dropWatchApiClient->sendUpdate($payload);
    }
    catch (\Exception $e) {
      $this->getLogger('dropwatch')->error($e->getMessage());
    }
  }

  /**
   * Check if the watchdog module is enabled.
   *
   * @return bool
   */
  public function checkIfDblogIsEnabled(): bool {
    $modules = $this->moduleHandler->getModuleList();
    if (array_key_exists('dblog', $modules)) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Formats a database log message.
   */
  private function getMessageDetails($row): array {
    $backtrace = '';
    $message = '';

    if (!empty($row->message) && !empty($row->variables)) {
      $variables = @unserialize($row->variables);

      if (!empty($variables)) {
        // Get the message.
        $message = str_replace('@backtrace_string', '', $row->message);
        $message = $this->t(Xss::filterAdmin($message), $variables);

        // Ensure backtrace strings are properly formatted.
        if (!empty($variables['@backtrace_string'])) {
          $backtrace = new FormattableMarkup(
            '<pre class="backtrace">@backtrace_string</pre>', $variables
          );
        }
      }
    }

    return [
      'message' => $message,
      'backtrace' => $backtrace,
    ];
  }
}
