<?php

namespace Drupal\dropwatch\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\dropwatch\Service\DropWatchService;
use Symfony\Component\DependencyInjection\ContainerInterface;

class DropWatchSettingsForm extends ConfigFormBase {

  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    protected DropWatchService $dropWatchService,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('dropwatch.service'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'dropwatch.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dropwatch_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dropwatch.settings');

    $form['declaration_container'] = [
      '#type' => 'fieldset',
    ];

    $form['declaration_container']['declaration'] = [
      '#type' => 'markup',
      '#markup' => '
        <h5>Declaration</h5>
        <p>No sensitive information about this site is ever sent to DropWatch. All information is sent securely through an API, protected by API Tokens.</p>
      ',
    ];

    $form['cron'] = [
      '#type' => 'fieldset',
    ];

    $form['cron']['information'] = [
      '#type' => 'markup',
      '#markup' => "
        <h5>Cron frequency</h5>
        <p>DropWatch will sync data every time cron runs. To achieve a more granular sync frequency, you may consider using <a href='https://www.drupal.org/project/ultimate_cron' title='Ultimate Cron'>Ultimate Cron</a>.</p>
      ",
    ];

    $form['site_url_container'] = [
      '#type' => 'fieldset',
    ];

    $form['site_url_container']['site_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Site URL'),
      '#description' => $this->t('This URL must match the site URL you entered into DropWatch.'),
      '#size' => 80,
      '#default_value' => $config->get('site_url'),
    ];

    $form['api_token_info_container'] = [
      '#type' => 'fieldset',
    ];

    $form['api_token_info_container']['token_info'] = [
      '#type' => 'markup',
      '#markup' => "
        <h5>API Token Information</h5>
        <p>You are required to create an API Token for each site you want to track with DropWatch.</p>
        <p>Once you have your API Token, it will need to be added to the settings.php file on your sites server. e.g</p>
      ",
    ];

    $form['api_token_info_container'][''] = [
      '#type' => 'textfield',
      '#value' => "\$settings['dropwatch_api_token'] = '<YOUR_DROPWATCH_SITE_API_TOKEN>';",
      '#size' => 80,
      '#prefix' => '<code>',
      '#suffix' => '</code>',
      '#attributes' => [
        'readonly' => 'readonly',
      ],
    ];

    $form['tracking'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('What would you like DropWatch to track?'),
    ];

    $form['tracking']['core'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Core'),
      '#description' => $this->t('Core version, warnings and general site details.'),
      '#default_value' => $config->get('core'),
    ];

    $form['tracking']['php'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('PHP'),
      '#description' => $this->t('PHP version, caching, memory limit and other PHP related details.'),
      '#default_value' => $config->get('php'),
    ];

    if ($this->dropWatchService->checkIfDblogIsEnabled()) {
      $form['tracking']['php_logs'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('PHP logs'),
        '#description' => $this->t('PHP errors and warnings from Watchdog. Requires dblog to be enabled.'),
        '#default_value' => $config->get('php_logs'),
      ];
    }

    $form['tracking']['web_server'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Web Server'),
      '#description' => $this->t('Web server version and other web server related details.'),
      '#default_value' => $config->get('web_server'),
    ];

    $form['tracking']['database'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Database'),
      '#description' => $this->t('Database version and other database related details.'),
      '#default_value' => $config->get('database'),
    ];

    $form['tracking']['contrib_modules'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Contrib modules'),
      '#description' => $this->t('Contrib modules installed and versions.'),
      '#default_value' => $config->get('contrib_modules'),
    ];

    $form['tracking']['contrib_themes'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Contrib themes'),
      '#description' => $this->t('Contrib themes installed and versions.'),
      '#default_value' => $config->get('contrib_themes'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $values = $form_state->getValues();
    $settings = $this->config('dropwatch.settings');
    $settings->set('site_url', $values['site_url']);
    $settings->set('core', $values['core']);
    $settings->set('php', $values['php']);
    $settings->set('web_server', $values['web_server']);
    $settings->set('database', $values['database']);
    $settings->set('contrib_modules', $values['contrib_modules']);
    $settings->set('contrib_themes', $values['contrib_themes']);
    $settings->set('php_logs', $values['php_logs']);
    $settings->save();
  }
}
