// postbuild-move.js
// Moves submodule JS files from main dist folder to their respective submodule dist folders after Vite build.

const fs = require('fs');
const path = require('path');
const glob = require('glob');
const rimraf = require('rimraf');

// Find all JS and CSS files in dist/modules/*
const jsFiles = glob.sync('dist/modules/*/dist/js/**/*.js');
const cssFiles = glob.sync('dist/modules/*/dist/css/**/*.css');

// Find all submodules that have JS or CSS files to move
const submodules = new Set();
jsFiles.forEach(file => {
  const match = file.match(/dist\/modules\/([^/]+)\/dist\/js\//);
  if (match) submodules.add(match[1]);
});
cssFiles.forEach(file => {
  const match = file.match(/dist\/modules\/([^/]+)\/dist\/css\//);
  if (match) submodules.add(match[1]);
});

// Helper to clear a folder
function clearFolder(folderPath, pattern) {
  if (fs.existsSync(folderPath)) {
    glob.sync(path.join(folderPath, pattern)).forEach(entry => {
      rimraf.sync(entry);
      console.log(`Deleted ${entry}`);
    });
  }
}

// Clear dist/js and dist/css for each submodule
submodules.forEach(submodule => {
  const jsDistPath = path.join('modules', submodule, 'dist', 'js');
  clearFolder(jsDistPath, '*');
  const cssDistPath = path.join('modules', submodule, 'dist', 'css');
  clearFolder(cssDistPath, '*');
});

// Helper to move files
function moveFile(file, destDir, destFile) {
  fs.mkdirSync(destDir, { recursive: true });
  // Delete all files in the destination directory before moving
  glob.sync(path.join(destDir, '*')).forEach(existingFile => {
    if (fs.existsSync(existingFile)) {
      fs.unlinkSync(existingFile);
      console.log(`Deleted existing file: ${existingFile}`);
    }
  });
  fs.renameSync(file, destFile);
  console.log(`Moved ${file} -> ${destFile}`);
}

// Move JS files
jsFiles.forEach(file => {
  const match = file.match(/dist\/modules\/([^/]+)\/dist\/js\/(.+)/);
  if (!match) return;
  const submodule = match[1];
  const jsPath = match[2];
  const destDir = path.join('modules', submodule, 'dist', 'js', path.dirname(jsPath));
  const destFile = path.join(destDir, path.basename(jsPath));
  moveFile(file, destDir, destFile);
});

cssFiles.forEach(file => {
  const match = file.match(/dist\/modules\/([^/]+)\/dist\/css\/(.+)/);
  if (!match) return;
  const submodule = match[1];
  const cssPath = match[2];
  const destDir = path.join('modules', submodule, 'dist', 'css', path.dirname(cssPath));
  const destFile = path.join(destDir, path.basename(cssPath));
  moveFile(file, destDir, destFile);
});

// Remove dist/modules folder after moving files
const distModulesPath = path.join('dist', 'modules');
if (fs.existsSync(distModulesPath)) {
  rimraf.sync(distModulesPath);
  console.log(`Removed ${distModulesPath}`);
}
