
// ===============================
// Vite Config for Main & Submodules
// ===============================

import { glob } from 'glob';
import path, { resolve } from 'path';
import { defineConfig } from 'vite';
import { viteStaticCopy } from 'vite-plugin-static-copy';

// === CONFIGURATION VARIABLES ===
// Set to true if you want to include submodules
const USE_SUBMODULES = true;
// CSS output style: 'expanded', 'compressed', etc.
const CSS_OUTPUT_STYLE = 'compressed';
// JS minification: true to minimize, false for readable output
const MINIFY_JS = false;
// Dummy file path for Vite to avoid errors when no JS files are found (Vite needs an entry point, which is index.html or a JS file)
const DUMMY_JS_FILE_PATH = 'vite.fallback_entrypoint.js';
// Main project folders
const MAIN_JS_DIR = 'js';
const MAIN_SCSS_DIR = 'scss';
const MAIN_FONTS_DIR = 'assets/fonts';

// Submodule folders (relative to project root)
const SUBMODULES_DIR = 'modules';
const SUBMODULE_JS_GLOB = `${SUBMODULES_DIR}/**/js/**/*.js`;
const SUBMODULE_SCSS_GLOB = `${SUBMODULES_DIR}/**/scss/**/[^_]*.scss`;
const SUBMODULE_FONTS_GLOB = `${SUBMODULES_DIR}/**/assets/fonts/**/*.{woff,woff2,ttf,otf,eot,svg}`;

// Import paths for Sass (node_modules + fonts)
const importPaths = ['node_modules', MAIN_FONTS_DIR];

// === Helper: Get files from glob patterns ===
function getFiles(patterns) {
  return patterns
    .map(pattern => glob.sync(pattern))
    .flat()
    .filter(file => !!file);
}

// === GLOB PATTERNS ===
const jsGlob = USE_SUBMODULES
  ? [`${MAIN_JS_DIR}/**/*.js`, SUBMODULE_JS_GLOB]
  : [`${MAIN_JS_DIR}/**/*.js`];
const scssGlob = USE_SUBMODULES
  ? [`${MAIN_SCSS_DIR}/**/[^_]*.scss`, SUBMODULE_SCSS_GLOB]
  : [`${MAIN_SCSS_DIR}/**/[^_]*.scss`];
const fontsGlob = USE_SUBMODULES
  ? [`${MAIN_FONTS_DIR}/**/*.{woff,woff2,ttf,otf,eot,svg}`, SUBMODULE_FONTS_GLOB]
  : [`${MAIN_FONTS_DIR}/**/*.{woff,woff2,ttf,otf,eot,svg}`];

// === JS ENTRY POINTS ===
const jsEntries = {};
getFiles(jsGlob).forEach(file => {
  jsEntries[file] = resolve(__dirname, file);
});
if (Object.keys(jsEntries).length === 0) {
  // If no JS files found, use a dummy file to avoid Vite errors
  jsEntries[DUMMY_JS_FILE_PATH] = DUMMY_JS_FILE_PATH;
}

// === VITE CONFIG ===
export default defineConfig({
  css: {
    preprocessorOptions: {
      scss: {
        includePaths: importPaths,
      },
    },
  },
  build: {
    outDir: 'dist',
    minify: MINIFY_JS ? 'esbuild' : false,
    rollupOptions: {
      input: jsEntries,
      output: {
        // Output JS files to their respective folders (main or submodule)
        entryFileNames: (chunkInfo) => {
          // chunkInfo.name is the full relative path from project root
          let filename;
          if (chunkInfo.name.startsWith('modules/')) {
            // modules/submodule/js/foo.js -> modules/submodule/dist/js/foo(.min).js
            filename = chunkInfo.name.replace(
              /(modules\/[^/]+)\/js\//,
              '$1/dist/js/'
            );
          } else if (chunkInfo.name.startsWith('js/')) {
            // js/foo.js -> js/foo(.min).js in main dist
            filename = `js/${chunkInfo.name.replace(/^js\//, '')}`;
          } else {
            // Fallback
            filename = chunkInfo.name;
          }
          // Add .min.js if MINIFY_JS is true
          if (MINIFY_JS) {
            return filename.replace(/\.js$/, '.min.js');
          }
          return filename;
        },
        assetFileNames: (assetInfo) => {
          return '[name][extname]';
        },
      },
    },
  },
  plugins: [
    viteStaticCopy({
      targets: [
        // === SCSS to CSS (main + submodules) ===
        ...getFiles(scssGlob).map(filename => {
          // If SCSS file is in a submodule, output to modules/<submodule>/dist/css/
          const submoduleMatch = filename.match(/^modules\/([^/]+)\/scss\/(.+)/);
          let dest = 'css';
          let rename;
          if (CSS_OUTPUT_STYLE === 'compressed') {
            rename = (filename) => filename + '.min.css';
          } else {
            rename = (filename) => filename + '.css';
          }
          if (submoduleMatch) {
            const submodule = submoduleMatch[1];
            const scssPath = submoduleMatch[2];
            dest = `modules/${submodule}/dist/css/${path.dirname(scssPath)}`;
            if (CSS_OUTPUT_STYLE === 'compressed') {
              rename = (filename) => path.basename(filename) + '.min.css';
            } else {
              rename = (filename) => path.basename(filename) + '.css';
            }
          }
          return {
            src: filename,
            dest,
            transform: (contents, filename) => {
              const { renderSync } = require('sass');
              const result = renderSync({
                file: filename,
                includePaths: importPaths,
                outputStyle: CSS_OUTPUT_STYLE,
              });
              return result.css.toString();
            },
            rename,
          };
        }),
        // === Fonts (main + submodules) ===
        ...getFiles(fontsGlob).map(fontFile => {
          const parentFolder = fontFile
            .replace(/^assets\/fonts\//, '')
            .replace(/^modules\//, '')
            .split('/')[0];
          return {
            src: fontFile,
            dest: `assets/fonts/${parentFolder}`,
          };
        }),
      ],
    }),
  ],
});
