<?php

namespace Drupal\druidfire\Plugin\Spell;

use Drupal\druidfire\SpellBase;

/**
 * Converts an entity reference revisions field to an entity reference field.
 *
 * @Spell(
 *   id = "err2er",
 *   label = @Translation("ERR to ER"),
 *   description = @Translation("Converts Entity Reference Revisions to Entity Reference fields."),
 * )
 */
class Err2Er extends SpellBase {

  /**
   * Modifies the schema of a field table.
   *
   * Removes the `target_revision_id` column and its associated index from the
   * schema.
   *
   * @param array $schema
   *   The current table schema.
   * @param string $tableName
   *   The name of the field table (e.g., node__field_name).
   * @param string $columnName
   *   The column of the field property being changed
   *   (e.g., field_name_target_id).
   * @param array $args
   *   Additional arguments required for the schema modification.
   *
   * @return array
   *   The modified table schema.
   */
  public function schema(array $schema, string $tableName, string $columnName, array $args = []): array {
    $columnName = str_replace('target_id', 'target_revision_id', $columnName);
    $this->schema->dropField($tableName, $columnName);
    unset($schema[$tableName]['fields'][$columnName]);
    unset($schema[$tableName]['indexes'][$columnName]);
    return $schema;
  }

  /**
   * Modifies the field storage configuration.
   *
   * Changes the field type to `entity_reference`.
   *
   * @param array $yaml
   *   The field storage configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified field storage configuration array.
   */
  public function storage(array $yaml, array $args = []): array {
    $yaml['type'] = 'entity_reference';
    return $yaml;
  }

  /**
   * Modifies the field configuration for a specific bundle.
   *
   * Configures the field to use the `entity_reference` field type.
   *
   * @param array $yaml
   *   The field configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified field configuration array.
   */
  public function field(array $yaml, array $args = []): array {
    $yaml['field_type'] = 'entity_reference';
    return $yaml;
  }

  /**
   * Modifies the form display configuration for a field.
   *
   * Updates the widget type to use `entity_reference` instead of
   * `entity_reference_revisions`.
   *
   * @param array $yaml
   *   The form display configuration array.
   * @param string $fieldName
   *   The name of the field.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified form display configuration array.
   */
  public function formDisplay(array $yaml, $fieldName, array $args = []): array {
    if (isset($yaml['content'][$fieldName]['type'])) {
      $yaml['content'][$fieldName]['type'] = str_replace(
        'entity_reference_revisions_',
        'entity_reference_',
        $yaml['content'][$fieldName]['type']
      );
    }
    return $yaml;
  }

  /**
   * Modifies the view display configuration for a field.
   *
   * Reuses the form display configuration logic for view displays.
   *
   * @param array $yaml
   *   The view display configuration array.
   * @param string $fieldName
   *   The name of the field.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified view display configuration array.
   */
  public function viewDisplay(array $yaml, $fieldName, array $args = []): array {
    return $this->formDisplay($yaml, $fieldName);
  }

}
