<?php

namespace Drupal\druidfire\Plugin\Spell;

use Drupal\druidfire\SpellBase;

/**
 * Converts a string field to a formatted text field.
 *
 * @Spell(
 *   id = "string2formatted",
 *   label = @Translation("String to Formatted Text"),
 *   description = @Translation("Converts a string field to a formatted text field."),
 * )
 */
class String2Formatted extends SpellBase {

  /**
   * Modifies the schema of a field table.
   *
   * Changes the field type to 'text' with a large size and adds a new column
   * for the text format. Updates the schema to include the new column and
   * its index.
   *
   * @param array $schema
   *   The current table schema.
   * @param string $tableName
   *   The name of the field table (e.g., node__body).
   * @param string $columnName
   *   The column of the field property being changed (e.g., body_value).
   * @param array $args
   *   Additional arguments required for the schema modification.
   *
   * @return array
   *   The modified table schema.
   */
  public function schema(array $schema, string $tableName, string $columnName, array $args = []): array {
    $spec = [
      'type' => 'text',
      'size' => 'big',
    ];
    $schema[$tableName]['fields'][$columnName] = $spec;
    $this->schema->changeField($tableName, $columnName, $columnName, $spec);
    $format = preg_replace('/_value$/', '_format', $columnName);
    $schema[$tableName]['fields'][$format] = [
      'type' => 'varchar_ascii',
      'length' => 255,
    ];
    $schema[$tableName]['indexes'][$format] = [$format];
    $this->schema->addField($tableName, $format, $schema[$tableName]['fields'][$format]);
    $this->schema->addIndex($tableName, $format, $schema[$tableName]['indexes'][$format], $schema[$tableName]);
    return $schema;
  }

  /**
   * Modifies the field storage configuration.
   *
   * Changes the field type to 'text_long'.
   *
   * @param array $yaml
   *   The field storage configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified field storage configuration array.
   */
  public function storage(array $yaml, array $args = []): array {
    $yaml['type'] = 'text_long';
    return $yaml;
  }

  /**
   * Modifies the field configuration for a specific bundle.
   *
   * Configures the field to use the 'text_long' field type.
   *
   * @param array $yaml
   *   The field configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified field configuration array.
   */
  public function field(array $yaml, array $args = []): array {
    $yaml['field_type'] = 'text_long';
    return $yaml;
  }

  /**
   * Modifies the form display configuration for a field.
   *
   * Sets the widget type to 'text_textarea' and adjusts the number of rows
   * displayed in the textarea.
   *
   * @param array $yaml
   *   The form display configuration array.
   * @param string $fieldName
   *   The name of the field.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified form display configuration array.
   */
  public function formDisplay(array $yaml, $fieldName, array $args = []): array {
    $yaml['content'][$fieldName]['type'] = 'text_textarea';
    unset($yaml['content'][$fieldName]['settings']['size']);
    $yaml['content'][$fieldName]['settings']['rows'] = 9;
    return $yaml;
  }

}
