<?php

namespace Drupal\druidfire\Plugin\Spell;

use Drupal\druidfire\SpellBase;
use Drupal\taxonomy\Entity\Term;

/**
 * Converts a string field to a taxonomy term reference field.
 *
 * @Spell(
 *   id = "string2taxonomyReference",
 *   label = @Translation("String to Taxonomy Reference"),
 *   description = @Translation("Converts a string field to a taxonomy term reference field."),
 * )
 */
class String2TaxonomyReference extends SpellBase {

  /**
   * Modifies the schema of a field table.
   *
   * Adds a new column for the taxonomy term reference, populates it with term
   * IDs based on the existing string values, and removes the original column.
   *
   * @param array $schema
   *   The current table schema.
   * @param string $tableName
   *   The name of the field table (e.g., node__body).
   * @param string $columnName
   *   The column of the field property being changed (e.g., body_value).
   * @param array $args
   *   Additional arguments required for the schema modification.
   *   - 'vid': The vocabulary ID to use for taxonomy terms.
   *
   * @return array
   *   The modified table schema.
   */
  public function schema(array $schema, string $tableName, string $columnName, array $args = []): array {
    $vid = $args['vid'] ?? '';
    if ($vid === '') {
      throw new \InvalidArgumentException('vid argument is required for string2taxonomyReference schema conversion.');
    }
    $new_column_name = preg_replace('/_value$/', '_target_id', $columnName);
    $new_column = [
      'description' => 'The ID of the target entity.',
      'type' => 'int',
      'unsigned' => TRUE,
    ];
    $this->schema->addField($tableName, $new_column_name, $new_column);
    $schema[$tableName]['fields'][$new_column_name] = $new_column;

    // Create terms from values if they don't already exist.
    $escapedColumnName = $this->database->escapeField($columnName);
    $subquery = $this->database->select('taxonomy_term_field_data', 't')
      ->fields('t')
      ->where("f.$escapedColumnName = t.name AND t.vid = :vid", [':vid' => $vid]);
    $query = $this->database->select($tableName, 'f');
    $query->distinct();
    $query->addField('f', $columnName, 'name');
    $query->addExpression(':vid', 'vid', [':vid' => $vid]);
    $query->notExists($subquery);
    array_map(fn($data) => Term::create($data)->save(), $query->execute()->fetchAll(\PDO::FETCH_ASSOC));

    // Populate the new column with term IDs.
    $update = $this->database->update($tableName);
    $update->expression($new_column_name, "(SELECT tid FROM {taxonomy_term_field_data} WHERE vid = :vid AND $escapedColumnName = name)", [':vid' => $vid]);
    $update->execute();
    $this->schema->dropField($tableName, $columnName);
    return $schema;
  }

  /**
   * Modifies the field storage configuration.
   *
   * Changes the field type to 'entity_reference' and sets the target type to
   * 'taxonomy_term'.
   *
   * @param array $yaml
   *   The field storage configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified field storage configuration array.
   */
  public function storage(array $yaml, array $args = []): array {
    $yaml['type'] = 'entity_reference';
    $yaml['settings'] = [];
    $yaml['settings']['target_type'] = 'taxonomy_term';
    return $yaml;
  }

  /**
   * Modifies the field configuration for a specific bundle.
   *
   * Configures the field to reference taxonomy terms.
   *
   * @param array $yaml
   *   The field configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *   - 'vid': The vocabulary ID to use for taxonomy terms.
   *
   * @return array
   *   The modified field configuration array.
   */
  public function field(array $yaml, array $args = []): array {
    $vid = $args['vid'] ?? '';
    $yaml['field_type'] = 'entity_reference';
    $yaml['settings'] = [];
    $yaml['settings']['handler'] = 'default:taxonomy_term';
    $yaml['settings']['handler_settings']['target_bundles'] = $vid ? [$vid => $vid] : [];
    $yaml['settings']['handler_settings']['auto_create'] = FALSE;
    return $yaml;
  }

  /**
   * Modifies the form display configuration for a field.
   *
   * Sets the widget type to 'entity_reference_label' for form displays.
   *
   * @param array $yaml
   *   The form display configuration array.
   * @param string $fieldName
   *   The name of the field.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified form display configuration array.
   */
  public function formDisplay(array $yaml, $fieldName, array $args = []): array {
    $yaml['content'][$fieldName]['type'] = 'entity_reference_label';
    $yaml['content'][$fieldName]['settings'] = [];
    $yaml['content'][$fieldName]['settings']['link'] = FALSE;
    return $yaml;
  }

  /**
   * Modifies the view display configuration for a field.
   *
   * Sets the widget type to 'options_select' for view displays.
   *
   * @param array $yaml
   *   The view display configuration array.
   * @param string $fieldName
   *   The name of the field.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified view display configuration array.
   */
  public function viewDisplay(array $yaml, $fieldName, array $args = []): array {
    $yaml['content'][$fieldName]['type'] = 'options_select';
    $yaml['content'][$fieldName]['settings'] = [];
    return $yaml;
  }

}
