<?php

namespace Drupal\Tests\druidfire\Unit;

use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Entity\EntityLastInstalledSchemaRepositoryInterface;
use Drupal\Core\Field\FieldTypePluginManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\druidfire\ConfigManager;
use Drupal\druidfire\SpellInterface;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for the ConfigManager service.
 *
 * @coversDefaultClass \Drupal\druidfire\ConfigManager
 * @group druidfire
 */
class ConfigManagerTest extends UnitTestCase {

  /**
   * The ConfigManager service under test.
   *
   * @var \Drupal\druidfire\ConfigManager
   */
  protected ConfigManager $configManager;

  /**
   * Mock configuration storage service.
   *
   * @var \Drupal\Core\Config\StorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected StorageInterface|MockObject $configStorage;

  /**
   * Mock field type plugin manager service.
   *
   * @var \Drupal\Core\Field\FieldTypePluginManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected FieldTypePluginManagerInterface|MockObject $fieldTypePluginManager;

  /**
   * Mock entity last installed schema repository service.
   *
   * @var \Drupal\Core\Entity\EntityLastInstalledSchemaRepositoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected EntityLastInstalledSchemaRepositoryInterface|MockObject $entityLastInstalledSchemaRepository;

  /**
   * Mock messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MessengerInterface|MockObject $messenger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->configStorage = $this->createMock(StorageInterface::class);
    $this->fieldTypePluginManager = $this->createMock(FieldTypePluginManagerInterface::class);
    $this->entityLastInstalledSchemaRepository = $this->createMock(EntityLastInstalledSchemaRepositoryInterface::class);
    $this->messenger = $this->createMock(MessengerInterface::class);

    $this->configManager = new ConfigManager(
      $this->configStorage,
      $this->fieldTypePluginManager,
      $this->entityLastInstalledSchemaRepository,
      $this->messenger
    );
  }

  /**
   * Tests changeConfig method for storage configuration.
   *
   * @covers ::changeConfig
   * @covers ::doChangeConfig
   */
  public function testChangeConfigForStorage(): void {
    $entityTypeId = 'node';
    $fieldName = 'body';
    $configType = 'storage';
    $optionalArguments = ['size' => 512];

    // Mock spell.
    $spell = $this->createMock(SpellInterface::class);
    // Mock config storage operations.
    $configName = "field.storage.$entityTypeId.$fieldName";
    $this->configStorage
      ->expects($this->once())
      ->method('listAll')
      ->with("field.$configType.$entityTypeId.")
      ->willReturn([$configName]);

    $configRecord = [
      'id' => "$entityTypeId.$fieldName",
      'field_name' => $fieldName,
      'entity_type' => $entityTypeId,
      'type' => 'text_long',
      'settings' => ['max_length' => 255],
    ];

    $this->configStorage
      ->expects($this->once())
      ->method('read')
      ->with($configName)
      ->willReturn($configRecord);

    $spell->expects($this->once())
      ->method('storage')
      ->with($configRecord, $optionalArguments)
      ->willReturn([
        'id' => "$entityTypeId.$fieldName",
        'field_name' => $fieldName,
        'entity_type' => $entityTypeId,
        'type' => 'text_long',
        'settings' => ['max_length' => 512],
      ]);

    $this->messenger
      ->expects($this->once())
      ->method('addMessage')
      ->with("Changing config $configName");

    $expectedRecord = [
      'id' => "$entityTypeId.$fieldName",
      'field_name' => $fieldName,
      'entity_type' => $entityTypeId,
      'type' => 'text_long',
      'settings' => ['max_length' => 512],
    ];

    $this->configStorage
      ->expects($this->once())
      ->method('write')
      ->with($configName, $expectedRecord);

    // For field storage configs, expect additional processing.
    $pluginClass = 'Drupal\text\Plugin\Field\FieldType\TextLongItem';
    $this->fieldTypePluginManager
      ->expects($this->once())
      ->method('getPluginClass')
      ->with('text_long')
      ->willReturn($pluginClass);

    // Verify the method calls.
    $this->entityLastInstalledSchemaRepository
      ->expects($this->once())
      ->method('setLastInstalledFieldStorageDefinition')
      ->with($this->isInstanceOf(FieldStorageConfig::class));

    $this->configManager->changeConfig($entityTypeId, $fieldName, $spell, $configType, $optionalArguments);
  }

  /**
   * Tests changeConfig method for field configuration with displays.
   *
   * @covers ::changeConfig
   * @covers ::doChangeConfig
   */
  public function testChangeConfigForFieldWithDisplays(): void {
    $entityTypeId = 'node';
    $fieldName = 'body';
    $configType = 'field';
    $optionalArguments = [];

    // Mock spell.
    $spell = $this->createMock(SpellInterface::class);
    $spell->expects($this->once())
      ->method('field')
      ->willReturn(['field_type' => 'text_long']);

    $spell->expects($this->once())
      ->method('formDisplay')
      ->willReturn(['content' => ['body' => ['type' => 'text_textarea']]]);

    $spell->expects($this->once())
      ->method('viewDisplay')
      ->willReturn(['content' => ['body' => ['type' => 'text_default']]]);

    // Mock field inspector clearing cached definitions.
    $fieldConfigName = "field.field.$entityTypeId.article.$fieldName";
    $formDisplayConfigName = "core.entity_form_display.$entityTypeId.article.default";
    $viewDisplayConfigName = "core.entity_view_display.$entityTypeId.article.default";

    $this->configStorage
      ->method('listAll')
      ->willReturnMap([
        ["field.$configType.$entityTypeId.", [$fieldConfigName]],
        ["core.entity_form_display.$entityTypeId.article.", [$formDisplayConfigName]],
        ["core.entity_view_display.$entityTypeId.article.", [$viewDisplayConfigName]],
      ]);

    $fieldConfig = ['field_type' => 'string'];
    $formDisplayConfig = ['content' => ['body' => ['type' => 'string_textfield']]];
    $viewDisplayConfig = ['content' => ['body' => ['type' => 'string']]];

    $this->configStorage
      ->method('read')
      ->willReturnMap([
        [$fieldConfigName, $fieldConfig],
        [$formDisplayConfigName, $formDisplayConfig],
        [$viewDisplayConfigName, $viewDisplayConfig],
      ]);

    $this->messenger
      ->expects($this->exactly(3))
      ->method('addMessage');

    $this->configStorage
      ->expects($this->exactly(3))
      ->method('write');

    $this->configManager->changeConfig($entityTypeId, $fieldName, $spell, $configType, $optionalArguments);
  }

  /**
   * Tests changeConfig with non-callable spell method.
   *
   * @covers ::changeConfig
   * @covers ::doChangeConfig
   */
  public function testChangeConfigWithNonCallableSpell(): void {
    $entityTypeId = 'node';
    $fieldName = 'body';
    $configType = 'storage';
    $optionalArguments = [];

    // Create a mock spell that doesn't have the storage method.
    $spell = $this->createMock(SpellInterface::class);
    $spell->method('storage')
      ->willReturn([]);

    $this->configStorage
      ->method('listAll')
      ->willReturn([]);

    // Should not call read or write since there are no config names.
    $this->configStorage
      ->expects($this->never())
      ->method('read');

    $this->configStorage
      ->expects($this->never())
      ->method('write');

    $this->configManager->changeConfig($entityTypeId, $fieldName, $spell, $configType, $optionalArguments);
  }

  /**
   * Tests doChangeConfig with missing config record.
   *
   * @covers ::doChangeConfig
   */
  public function testDoChangeConfigWithMissingRecord(): void {
    $entityTypeId = 'node';
    $fieldName = 'missing_field';
    $configType = 'storage';
    $optionalArguments = [];

    $spell = $this->createMock(SpellInterface::class);
    // Should not be called since config doesn't exist.
    $spell->expects($this->never())
      ->method('storage');

    $configName = "field.storage.$entityTypeId.$fieldName";
    $this->configStorage
      ->method('listAll')
      ->willReturn([$configName]);

    $this->configStorage
      ->method('read')
      ->with($configName)
      ->willReturn(NULL);

    $this->messenger
      ->expects($this->never())
      ->method('addMessage');

    $this->configStorage
      ->expects($this->never())
      ->method('write');

    $this->configManager->changeConfig($entityTypeId, $fieldName, $spell, $configType, $optionalArguments);
  }

  /**
   * Tests doChangeConfig with form display config prefix.
   *
   * @covers ::doChangeConfig
   */
  public function testDoChangeConfigWithFormDisplayPrefix(): void {
    $entityTypeId = 'node';
    $fieldName = 'body';
    $configType = 'field';
    $optionalArguments = [];

    $spell = $this->createMock(SpellInterface::class);
    $spell->method('field')
      ->willReturn(['field_type' => 'text_long']);

    $spell->expects($this->once())
      ->method('formDisplay')
      ->willReturn(['content' => ['body' => ['type' => 'text_textarea']]]);

    $spell->method('viewDisplay')
      ->willReturn(['content' => []]);

    // Mock field config.
    $fieldConfigName = "field.field.$entityTypeId.article.$fieldName";
    $this->configStorage
      ->method('listAll')
      ->willReturnMap([
        ["field.$configType.$entityTypeId.", [$fieldConfigName]],
        ["core.entity_form_display.$entityTypeId.article.", ["core.entity_form_display.$entityTypeId.article.default"]],
        ["core.entity_view_display.$entityTypeId.article.", ["core.entity_view_display.$entityTypeId.article.default"]],
      ]);

    $fieldConfig = ['field_type' => 'string'];
    $this->configStorage
      ->method('read')
      ->willReturnMap([
        [$fieldConfigName, $fieldConfig],
        ["core.entity_form_display.$entityTypeId.article.default", ['content' => []]],
        ["core.entity_view_display.$entityTypeId.article.default", ['content' => []]],
      ]);

    $this->configStorage
      ->expects($this->exactly(3))
      ->method('write');

    $this->configManager->changeConfig($entityTypeId, $fieldName, $spell, $configType, $optionalArguments);
  }

  /**
   * Tests changeConfig preserves field name in display configs.
   *
   * @covers ::changeConfig
   */
  public function testChangeConfigPreservesFieldNameInDisplays(): void {
    $entityTypeId = 'node';
    $fieldName = 'custom_field';
    $configType = 'field';
    $optionalArguments = ['test' => 'value'];

    $spell = $this->createMock(SpellInterface::class);
    $spell->method('field')->willReturn([]);
    $spell->method('formDisplay')->willReturn(['updated' => 'form']);
    $spell->method('viewDisplay')->willReturn(['updated' => 'view']);

    $this->configStorage
      ->method('listAll')
      ->willReturnMap([
        ["field.field.$entityTypeId.", ["field.field.$entityTypeId.bundle.$fieldName"]],
        ["core.entity_form_display.$entityTypeId.bundle.", ["core.entity_form_display.$entityTypeId.bundle.default"]],
        ["core.entity_view_display.$entityTypeId.bundle.", ["core.entity_view_display.$entityTypeId.bundle.default"]],
      ]);

    $this->configStorage
      ->method('read')
      ->willReturn(['existing' => 'config']);

    // The key test: formDisplay should be called with fieldName as first arg.
    $spell->expects($this->once())
      ->method('formDisplay')
      ->with(['existing' => 'config'], $fieldName, array_merge([$fieldName], $optionalArguments));

    // viewDisplay should be called with fieldName as first arg.
    $spell->expects($this->once())
      ->method('viewDisplay')
      ->with(['existing' => 'config'], $fieldName, array_merge([$fieldName], $optionalArguments));

    $this->configStorage->method('write')->willReturn(NULL);

    $this->configManager->changeConfig($entityTypeId, $fieldName, $spell, $configType, $optionalArguments);
  }

}
