<?php

namespace Drupal\Tests\druidfire\Unit;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Sql\DefaultTableMapping;
use Drupal\Core\Entity\Sql\SqlContentEntityStorage;
use Drupal\Core\Entity\Sql\SqlEntityStorageInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\druidfire\FieldInspector;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for the FieldInspector service.
 *
 * @coversDefaultClass \Drupal\druidfire\FieldInspector
 * @group druidfire
 */
class FieldInspectorTest extends UnitTestCase {

  /**
   * The FieldInspector service under test.
   *
   * @var \Drupal\druidfire\FieldInspector
   */
  protected FieldInspector $fieldInspector;

  /**
   * Mock entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected EntityTypeManagerInterface|MockObject $entityTypeManager;

  /**
   * Mock entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected EntityFieldManagerInterface|MockObject $entityFieldManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->entityFieldManager = $this->createMock(EntityFieldManagerInterface::class);

    $this->fieldInspector = new FieldInspector(
      $this->entityTypeManager,
      $this->entityFieldManager
    );
  }

  /**
   * Tests getColumnName with valid field and default property.
   *
   * @covers ::getColumnName
   */
  public function testGetColumnNameWithValidFieldAndDefaultProperty(): void {
    $entityTypeId = 'node';
    $fieldName = 'body';
    $propertyName = NULL;
    $expectedColumnName = 'body_value';

    // Mock entity storage and dependencies.
    $entityStorage = $this->createMock(SqlContentEntityStorage::class);
    $tableMapping = $this->createMock(DefaultTableMapping::class);
    $fieldDefinition = $this->createMock(FieldStorageDefinitionInterface::class);

    $this->entityTypeManager
      ->expects($this->once())
      ->method('getStorage')
      ->with($entityTypeId)
      ->willReturn($entityStorage);

    $entityStorage
      ->expects($this->once())
      ->method('getTableMapping')
      ->willReturn($tableMapping);

    $this->entityFieldManager
      ->expects($this->once())
      ->method('getFieldStorageDefinitions')
      ->with($entityTypeId)
      ->willReturn([$fieldName => $fieldDefinition]);

    $fieldDefinition
      ->expects($this->once())
      ->method('getMainPropertyName')
      ->willReturn('value');

    $tableMapping
      ->expects($this->once())
      ->method('getFieldColumnName')
      ->with($fieldDefinition, 'value')
      ->willReturn($expectedColumnName);

    $result = $this->fieldInspector->getColumnName($entityTypeId, $fieldName, $propertyName);

    $this->assertEquals($expectedColumnName, $result);
  }

  /**
   * Tests getColumnName with valid field and specific property.
   *
   * @covers ::getColumnName
   */
  public function testGetColumnNameWithValidFieldAndSpecificProperty(): void {
    $entityTypeId = 'node';
    $fieldName = 'body';
    $propertyName = 'format';
    $expectedColumnName = 'body_format';

    // Mock entity storage and dependencies.
    $entityStorage = $this->createMock(SqlContentEntityStorage::class);
    $tableMapping = $this->createMock(DefaultTableMapping::class);
    $fieldDefinition = $this->createMock(FieldStorageDefinitionInterface::class);

    $this->entityTypeManager
      ->method('getStorage')
      ->willReturn($entityStorage);

    $entityStorage
      ->method('getTableMapping')
      ->willReturn($tableMapping);

    $this->entityFieldManager
      ->method('getFieldStorageDefinitions')
      ->willReturn([$fieldName => $fieldDefinition]);

    // Should not call getMainPropertyName since we provided a property.
    $fieldDefinition
      ->expects($this->never())
      ->method('getMainPropertyName');

    $tableMapping
      ->expects($this->once())
      ->method('getFieldColumnName')
      ->with($fieldDefinition, $propertyName)
      ->willReturn($expectedColumnName);

    $result = $this->fieldInspector->getColumnName($entityTypeId, $fieldName, $propertyName);

    $this->assertEquals($expectedColumnName, $result);
  }

  /**
   * Tests getColumnName with empty string property name (should use default).
   *
   * @covers ::getColumnName
   */
  public function testGetColumnNameWithEmptyStringProperty(): void {
    $entityTypeId = 'node';
    $fieldName = 'title';
    $propertyName = '';
    $expectedColumnName = 'title_value';

    // Mock entity storage and dependencies.
    $entityStorage = $this->createMock(SqlContentEntityStorage::class);
    $tableMapping = $this->createMock(DefaultTableMapping::class);
    $fieldDefinition = $this->createMock(FieldStorageDefinitionInterface::class);

    $this->entityTypeManager
      ->method('getStorage')
      ->willReturn($entityStorage);

    $entityStorage
      ->method('getTableMapping')
      ->willReturn($tableMapping);

    $this->entityFieldManager
      ->method('getFieldStorageDefinitions')
      ->willReturn([$fieldName => $fieldDefinition]);

    $fieldDefinition
      ->expects($this->once())
      ->method('getMainPropertyName')
      ->willReturn('value');

    $tableMapping
      ->method('getFieldColumnName')
      ->willReturn($expectedColumnName);

    $result = $this->fieldInspector->getColumnName($entityTypeId, $fieldName, $propertyName);

    $this->assertEquals($expectedColumnName, $result);
  }

  /**
   * Tests getColumnName with nonexistent field.
   *
   * @covers ::getColumnName
   */
  public function testGetColumnNameWithNonexistentField(): void {
    $entityTypeId = 'node';
    $fieldName = 'nonexistent_field';
    $propertyName = NULL;

    $entityStorage = $this->createMock(SqlContentEntityStorage::class);
    $tableMapping = $this->createMock(DefaultTableMapping::class);

    $this->entityTypeManager
      ->method('getStorage')
      ->willReturn($entityStorage);

    $entityStorage
      ->method('getTableMapping')
      ->willReturn($tableMapping);

    $this->entityFieldManager
      ->method('getFieldStorageDefinitions')
      ->willReturn([]);

    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Can only change default SQL fields.');

    $this->fieldInspector->getColumnName($entityTypeId, $fieldName, $propertyName);
  }

  /**
   * Tests getColumnName with non-SQL entity storage.
   *
   * @covers ::getColumnName
   */
  public function testGetColumnNameWithNonSqlStorage(): void {
    $entityTypeId = 'config_entity';
    $fieldName = 'some_field';
    $propertyName = NULL;

    // Mock non-SQL entity storage.
    $entityStorage = $this->createMock(SqlEntityStorageInterface::class);

    $this->entityTypeManager
      ->method('getStorage')
      ->willReturn($entityStorage);

    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Can only change default SQL fields.');

    $this->fieldInspector->getColumnName($entityTypeId, $fieldName, $propertyName);
  }

  /**
   * Tests getColumnName with non-default table mapping.
   *
   * @covers ::getColumnName
   */
  public function testGetColumnNameWithNonDefaultTableMapping(): void {
    $entityTypeId = 'node';
    $fieldName = 'body';
    $propertyName = NULL;

    // Mock entity storage with non-default table mapping.
    $entityStorage = $this->createMock(SqlContentEntityStorage::class);
    $tableMapping = $this->createMock(SqlEntityStorageInterface::class);
    $fieldDefinition = $this->createMock(FieldStorageDefinitionInterface::class);

    $this->entityTypeManager
      ->method('getStorage')
      ->willReturn($entityStorage);

    $entityStorage
      ->method('getTableMapping')
      ->willReturn($tableMapping);

    $this->entityFieldManager
      ->method('getFieldStorageDefinitions')
      ->willReturn([$fieldName => $fieldDefinition]);

    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Can only change default SQL fields.');

    $this->fieldInspector->getColumnName($entityTypeId, $fieldName, $propertyName);
  }

  /**
   * Tests clearCachedDefinitions method.
   *
   * @covers ::clearCachedDefinitions
   */
  public function testClearCachedDefinitions(): void {
    $this->entityTypeManager
      ->expects($this->once())
      ->method('clearCachedDefinitions');

    $this->fieldInspector->clearCachedDefinitions();
  }

}
