<?php

namespace Drupal\Tests\druidfire\Unit\Plugin\Spell;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Schema;
use Drupal\druidfire\Plugin\Spell\Err2Bricks;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for the Err2Bricks spell plugin.
 *
 * @coversDefaultClass \Drupal\druidfire\Plugin\Spell\Err2Bricks
 * @group druidfire
 */
class Err2BricksTest extends UnitTestCase {

  /**
   * The Err2Bricks spell plugin under test.
   *
   * @var \Drupal\druidfire\Plugin\Spell\Err2Bricks
   */
  protected Err2Bricks $err2Bricks;

  /**
   * Mock database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Connection|MockObject $database;

  /**
   * Mock database schema.
   *
   * @var \Drupal\Core\Database\Schema|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Schema|MockObject $schema;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->database = $this->createMock(Connection::class);
    $this->schema = $this->createMock(Schema::class);

    $this->database
      ->method('schema')
      ->willReturn($this->schema);

    $this->err2Bricks = new Err2Bricks([], 'err2bricks', [], $this->database);
  }

  /**
   * Tests schema modification adds bricks columns.
   *
   * @covers ::schema
   */
  public function testSchemaAddsBricksColumns(): void {
    $schema = [
      'node__field_paragraphs' => [
        'fields' => [
          'field_paragraphs_target_id' => [
            'type' => 'int',
            'unsigned' => TRUE,
            'not null' => FALSE,
          ],
          'field_paragraphs_target_revision_id' => [
            'type' => 'int',
            'unsigned' => TRUE,
            'not null' => FALSE,
          ],
        ],
      ],
    ];

    $tableName = 'node__field_paragraphs';
    $columnName = 'field_paragraphs_target_id';
    $args = [];

    // Expect depth and options columns to be added.
    $this->schema
      ->expects($this->exactly(2))
      ->method('addField')
      ->willReturnCallback(function ($table, $column, $spec) use ($tableName) {
        $this->assertEquals($tableName, $table);
        if ($column === 'field_paragraphs_depth') {
          $this->assertEquals([
            'type' => 'int',
            'size' => 'tiny',
            'unsigned' => TRUE,
            'not null' => FALSE,
          ], $spec);
        }
        elseif ($column === 'field_paragraphs_options') {
          $this->assertEquals([
            'type' => 'blob',
            'size' => 'normal',
            'not null' => FALSE,
            'serialize' => TRUE,
          ], $spec);
        }
        else {
          $this->fail("Unexpected column: $column");
        }
      });

    $result = $this->err2Bricks->schema($schema, $tableName, $columnName, $args);

    $expected = [
      'node__field_paragraphs' => [
        'fields' => [
          'field_paragraphs_target_id' => [
            'type' => 'int',
            'unsigned' => TRUE,
            'not null' => FALSE,
          ],
          'field_paragraphs_target_revision_id' => [
            'type' => 'int',
            'unsigned' => TRUE,
            'not null' => FALSE,
          ],
          'field_paragraphs_depth' => [
            'type' => 'int',
            'size' => 'tiny',
            'unsigned' => TRUE,
            'not null' => FALSE,
          ],
          'field_paragraphs_options' => [
            'type' => 'blob',
            'size' => 'normal',
            'not null' => FALSE,
            'serialize' => TRUE,
          ],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests storage configuration changes to bricks revisioned.
   *
   * @covers ::storage
   */
  public function testStorageChangesToBricksRevisioned(): void {
    $yaml = [
      'type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'paragraph',
      ],
    ];

    $args = [];

    $result = $this->err2Bricks->storage($yaml, $args);

    $expected = [
      'type' => 'bricks_revisioned',
      'settings' => [
        'target_type' => 'paragraph',
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests field configuration changes to bricks revisioned.
   *
   * @covers ::field
   */
  public function testFieldChangesToBricksRevisioned(): void {
    $yaml = [
      'field_type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'paragraph',
        'handler' => 'default:paragraph',
      ],
    ];

    $args = [];

    $result = $this->err2Bricks->field($yaml, $args);

    $expected = [
      'field_type' => 'bricks_revisioned',
      'settings' => [
        'target_type' => 'paragraph',
        'handler' => 'default:paragraph',
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests form display leaves configuration unchanged.
   *
   * @covers ::formDisplay
   */
  public function testFormDisplayLeavesConfigUnchanged(): void {
    $yaml = [
      'content' => [
        'field_paragraphs' => [
          'type' => 'entity_reference_revisions_autocomplete',
          'settings' => [
            'match_operator' => 'CONTAINS',
          ],
        ],
      ],
    ];

    $fieldName = 'field_paragraphs';
    $args = [];

    $result = $this->err2Bricks->formDisplay($yaml, $fieldName, $args);

    // Should return unchanged since it uses parent implementation.
    $this->assertEquals($yaml, $result);
  }

  /**
   * Tests view display updates widget type to bricks_nested.
   *
   * @covers ::viewDisplay
   */
  public function testViewDisplayUpdatesWidgetType(): void {
    $yaml = [
      'content' => [
        'field_paragraphs' => [
          'type' => 'entity_reference_revisions_entity_view',
          'settings' => [
            'view_mode' => 'default',
          ],
        ],
      ],
    ];

    $fieldName = 'field_paragraphs';
    $args = [];

    $result = $this->err2Bricks->viewDisplay($yaml, $fieldName, $args);

    $expected = [
      'content' => [
        'field_paragraphs' => [
          'type' => 'bricks_revisions_nested',
          'settings' => [
            'view_mode' => 'default',
          ],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests view display with missing field leaves config unchanged.
   *
   * @covers ::viewDisplay
   */
  public function testViewDisplayWithMissingField(): void {
    $yaml = [
      'content' => [
        'other_field' => [
          'type' => 'string',
        ],
      ],
    ];

    $fieldName = 'field_paragraphs';
    $args = [];

    $result = $this->err2Bricks->viewDisplay($yaml, $fieldName, $args);

    // Should return unchanged since field is not in content.
    $this->assertEquals($yaml, $result);
  }

}
