<?php

namespace Drupal\Tests\druidfire\Unit\Plugin\Spell;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Schema;
use Drupal\druidfire\Plugin\Spell\Err2Er;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for the Err2Er spell plugin.
 *
 * @coversDefaultClass \Drupal\druidfire\Plugin\Spell\Err2Er
 * @group druidfire
 */
class Err2ErTest extends UnitTestCase {

  /**
   * The Err2Er spell plugin under test.
   *
   * @var \Drupal\druidfire\Plugin\Spell\Err2Er
   */
  protected Err2Er $err2Er;

  /**
   * Mock database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Connection|MockObject $database;

  /**
   * Mock database schema.
   *
   * @var \Drupal\Core\Database\Schema|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Schema|MockObject $schema;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->database = $this->createMock(Connection::class);
    $this->schema = $this->createMock(Schema::class);

    $this->database
      ->method('schema')
      ->willReturn($this->schema);

    $this->err2Er = new Err2Er([], 'err2er', [], $this->database);
  }

  /**
   * Tests schema modification removes revision column.
   *
   * @covers ::schema
   */
  public function testSchemaRemovesRevisionColumn(): void {
    $schema = [
      'node__field_paragraphs' => [
        'fields' => [
          'field_paragraphs_target_id' => [
            'type' => 'int',
            'unsigned' => TRUE,
            'not null' => FALSE,
          ],
          'field_paragraphs_target_revision_id' => [
            'type' => 'int',
            'unsigned' => TRUE,
            'not null' => FALSE,
          ],
        ],
      ],
    ];

    $tableName = 'node__field_paragraphs';
    $columnName = 'field_paragraphs_target_id';
    $args = [];

    // Expect the revision column to be dropped.
    $this->schema
      ->expects($this->once())
      ->method('dropField')
      ->with($tableName, 'field_paragraphs_target_revision_id');

    $result = $this->err2Er->schema($schema, $tableName, $columnName, $args);

    $expected = [
      'node__field_paragraphs' => [
        'fields' => [
          'field_paragraphs_target_id' => [
            'type' => 'int',
            'unsigned' => TRUE,
            'not null' => FALSE,
          ],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests storage configuration changes to entity_reference.
   *
   * @covers ::storage
   */
  public function testStorageChangesToEntityReference(): void {
    $yaml = [
      'type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'paragraph',
      ],
    ];

    $args = [];

    $result = $this->err2Er->storage($yaml, $args);

    $expected = [
      'type' => 'entity_reference',
      'settings' => [
        'target_type' => 'paragraph',
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests field configuration changes to entity_reference.
   *
   * @covers ::field
   */
  public function testFieldChangesToEntityReference(): void {
    $yaml = [
      'field_type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'paragraph',
        'handler' => 'default:paragraph',
      ],
    ];

    $args = [];

    $result = $this->err2Er->field($yaml, $args);

    $expected = [
      'field_type' => 'entity_reference',
      'settings' => [
        'target_type' => 'paragraph',
        'handler' => 'default:paragraph',
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests form display updates widget type.
   *
   * @covers ::formDisplay
   */
  public function testFormDisplayUpdatesWidgetType(): void {
    $yaml = [
      'content' => [
        'field_paragraphs' => [
          'type' => 'entity_reference_revisions_autocomplete',
          'settings' => [
            'match_operator' => 'CONTAINS',
          ],
        ],
      ],
    ];

    $fieldName = 'field_paragraphs';
    $args = [];

    $result = $this->err2Er->formDisplay($yaml, $fieldName, $args);

    $expected = [
      'content' => [
        'field_paragraphs' => [
          'type' => 'entity_reference_autocomplete',
          'settings' => [
            'match_operator' => 'CONTAINS',
          ],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests view display delegates to form display.
   *
   * @covers ::viewDisplay
   */
  public function testViewDisplayDelegatesToFormDisplay(): void {
    $yaml = [
      'content' => [
        'field_paragraphs' => [
          'type' => 'entity_reference_revisions_entity_view',
          'settings' => [
            'view_mode' => 'default',
          ],
        ],
      ],
    ];

    $fieldName = 'field_paragraphs';
    $args = [];

    $result = $this->err2Er->viewDisplay($yaml, $fieldName, $args);

    // Should call formDisplay method, which updates the widget type.
    $expected = [
      'content' => [
        'field_paragraphs' => [
          'type' => 'entity_reference_entity_view',
          'settings' => [
            'view_mode' => 'default',
          ],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests form display with missing field leaves config unchanged.
   *
   * @covers ::formDisplay
   */
  public function testFormDisplayWithMissingField(): void {
    $yaml = [
      'content' => [
        'other_field' => [
          'type' => 'string_textfield',
        ],
      ],
    ];

    $fieldName = 'field_paragraphs';
    $args = [];

    $result = $this->err2Er->formDisplay($yaml, $fieldName, $args);

    // Should return unchanged since field is not in content.
    $this->assertEquals($yaml, $result);
  }

}
