<?php

namespace Drupal\Tests\druidfire\Unit\Plugin\Spell;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Schema;
use Drupal\druidfire\Plugin\Spell\String2Formatted;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for the String2Formatted spell plugin.
 *
 * @coversDefaultClass \Drupal\druidfire\Plugin\Spell\String2Formatted
 * @group druidfire
 */
class String2FormattedTest extends UnitTestCase {

  /**
   * The String2Formatted spell plugin under test.
   *
   * @var \Drupal\druidfire\Plugin\Spell\String2Formatted
   */
  protected String2Formatted $string2Formatted;

  /**
   * Mock database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Connection|MockObject $database;

  /**
   * Mock database schema.
   *
   * @var \Drupal\Core\Database\Schema|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Schema|MockObject $schema;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->database = $this->createMock(Connection::class);
    $this->schema = $this->createMock(Schema::class);

    $this->database
      ->method('schema')
      ->willReturn($this->schema);

    $this->string2Formatted = new String2Formatted([], 'string2formatted', [], $this->database);
  }

  /**
   * Tests schema modification converts string to text with format column.
   *
   * @covers ::schema
   */
  public function testSchemaConvertsStringToText(): void {
    $schema = [
      'node__body' => [
        'fields' => [
          'body_value' => [
            'type' => 'varchar',
            'length' => 255,
            'not null' => FALSE,
          ],
        ],
      ],
    ];

    $tableName = 'node__body';
    $columnName = 'body_value';
    $args = [];

    // Expect the value column to be changed to text.
    $this->schema
      ->expects($this->once())
      ->method('changeField')
      ->with(
        $tableName,
        $columnName,
        $columnName,
        ['type' => 'text', 'size' => 'big']
      );

    // Expect the format column to be added.
    $this->schema
      ->expects($this->once())
      ->method('addField')
      ->with(
        $tableName,
        'body_format',
        [
          'type' => 'varchar_ascii',
          'length' => 255,
        ]
      );

    // Expect index to be added.
    $this->schema
      ->expects($this->once())
      ->method('addIndex')
      ->with(
        $tableName,
        'body_format',
        ['body_format'],
        $this->callback(fn($value) => is_array($value))
      );

    $result = $this->string2Formatted->schema($schema, $tableName, $columnName, $args);

    $expected = [
      'node__body' => [
        'fields' => [
          'body_value' => [
            'type' => 'text',
            'size' => 'big',
          ],
          'body_format' => [
            'type' => 'varchar_ascii',
            'length' => 255,
          ],
        ],
        'indexes' => [
          'body_format' => ['body_format'],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests storage configuration changes to text_long.
   *
   * @covers ::storage
   */
  public function testStorageChangesToTextLong(): void {
    $yaml = [
      'type' => 'string',
      'settings' => [
        'max_length' => 255,
        'case_sensitive' => FALSE,
      ],
    ];

    $args = [];

    $result = $this->string2Formatted->storage($yaml, $args);

    $expected = [
      'type' => 'text_long',
      'settings' => [
        'max_length' => 255,
        'case_sensitive' => FALSE,
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests field configuration changes to text_long.
   *
   * @covers ::field
   */
  public function testFieldChangesToTextLong(): void {
    $yaml = [
      'field_type' => 'string',
      'settings' => [
        'max_length' => 255,
      ],
    ];

    $args = [];

    $result = $this->string2Formatted->field($yaml, $args);

    $expected = [
      'field_type' => 'text_long',
      'settings' => [
        'max_length' => 255,
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests form display updates widget to text_textarea.
   *
   * @covers ::formDisplay
   */
  public function testFormDisplayUpdatesWidget(): void {
    $yaml = [
      'content' => [
        'body' => [
          'type' => 'string_textfield',
          'settings' => [
            'size' => 60,
          ],
        ],
      ],
    ];

    $fieldName = 'body';
    $args = [];

    $result = $this->string2Formatted->formDisplay($yaml, $fieldName, $args);

    $expected = [
      'content' => [
        'body' => [
          'type' => 'text_textarea',
          'settings' => [
            'rows' => 9,
          ],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests view display configuration defaults to parent implementation.
   *
   * @covers ::viewDisplay
   */
  public function testViewDisplayUsesDefault(): void {
    $yaml = [
      'content' => [
        'body' => [
          'type' => 'string',
          'settings' => [],
        ],
      ],
    ];

    $fieldName = 'body';
    $args = [];

    $result = $this->string2Formatted->viewDisplay($yaml, $fieldName, $args);

    // Should return unchanged since it uses parent implementation.
    $this->assertEquals($yaml, $result);
  }

}
