<?php

namespace Drupal\Tests\druidfire\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Schema;
use Drupal\druidfire\SpellBase;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for the SpellBase class.
 *
 * @coversDefaultClass \Drupal\druidfire\SpellBase
 * @group druidfire
 */
class SpellBaseTest extends UnitTestCase {

  /**
   * The SpellBase instance under test.
   *
   * @var \Drupal\druidfire\SpellBase
   */
  protected SpellBase $spellBase;

  /**
   * Mock database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Connection|MockObject $database;

  /**
   * Mock database schema service.
   *
   * @var \Drupal\Core\Database\Schema|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Schema|MockObject $schema;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->database = $this->createMock(Connection::class);
    $this->schema = $this->createMock(Schema::class);

    $this->database
      ->method('schema')
      ->willReturn($this->schema);

    $this->spellBase = new SpellBase([], 'test_spell', [], $this->database);
  }

  /**
   * Tests the schema method default implementation.
   *
   * @covers ::schema
   */
  public function testSchemaDefault(): void {
    $schema = [
      'test_table' => [
        'fields' => [
          'test_field' => [
            'type' => 'varchar',
            'length' => 255,
          ],
        ],
      ],
    ];

    $result = $this->spellBase->schema($schema, 'test_table', 'test_field');
    $this->assertSame($schema, $result);
  }

  /**
   * Tests the storage method default implementation.
   *
   * @covers ::storage
   */
  public function testStorageDefault(): void {
    $yaml = [
      'type' => 'string',
      'settings' => [
        'max_length' => 255,
      ],
    ];

    $result = $this->spellBase->storage($yaml);
    $this->assertSame($yaml, $result);
  }

  /**
   * Tests the field method default implementation.
   *
   * @covers ::field
   */
  public function testFieldDefault(): void {
    $yaml = [
      'field_type' => 'string',
      'settings' => [],
    ];

    $result = $this->spellBase->field($yaml);
    $this->assertSame($yaml, $result);
  }

  /**
   * Tests the formDisplay method default implementation.
   *
   * @covers ::formDisplay
   */
  public function testFormDisplayDefault(): void {
    $yaml = [
      'content' => [
        'test_field' => [
          'type' => 'string_textfield',
          'settings' => [
            'size' => 60,
          ],
        ],
      ],
    ];

    $result = $this->spellBase->formDisplay($yaml, 'test_field');
    $this->assertSame($yaml, $result);
  }

  /**
   * Tests the viewDisplay method default implementation.
   *
   * @covers ::viewDisplay
   */
  public function testViewDisplayDefault(): void {
    $yaml = [
      'content' => [
        'test_field' => [
          'type' => 'string',
          'settings' => [],
        ],
      ],
    ];

    $result = $this->spellBase->viewDisplay($yaml, 'test_field');
    $this->assertSame($yaml, $result);
  }

  /**
   * Tests constructor initializes schema service.
   *
   * @covers ::__construct
   */
  public function testConstructor(): void {
    // Use reflection to access protected schema property.
    $reflection = new \ReflectionClass($this->spellBase);
    $schemaProperty = $reflection->getProperty('schema');
    $schemaProperty->setAccessible(TRUE);
    $schema = $schemaProperty->getValue($this->spellBase);

    $this->assertSame($this->schema, $schema);
  }

}
