<?php

namespace Drupal\druidfire\Commands;

use Drupal\druidfire\Druidfire;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Helper\Table;

/**
 * Provides Drush commands for the Druidfire module.
 *
 * This class defines custom Drush commands to modify entity field properties
 * dynamically, such as resizing fields, converting string fields to formatted
 * fields, and changing entity reference fields.
 */
class DruidfireCommands extends DrushCommands {

  /**
   * Constructs a new DruidfireCommands object.
   *
   * @param \Drupal\druidfire\Druidfire $druidfire
   *   The Druidfire service.
   */
  public function __construct(
    protected Druidfire $druidfire,
  ) {
    parent::__construct();
  }

  /**
   * Resizes a field to a new length.
   *
   * @param string $entityTypeId
   *   The entity type ID (e.g., 'node', 'user').
   * @param string $fieldName
   *   The name of the field to resize. Best used for text fields.
   * @param int $size
   *   The new size for the field.
   * @param string|null $property
   *   (Optional) The specific property of the field to resize.
   *
   * @command druidfire:resize
   *
   * @usage drush druidfire:resize node body 255
   *   Resizes the 'body' field of the 'node' entity type to 255 characters.
   */
  public function resize($entityTypeId, $fieldName, $size, $property = NULL) {
    // @phpstan-ignore-next-line
    $this->druidfire->resize($entityTypeId, $fieldName, ['size' => $size, 'property' => $property]);
  }

  /**
   * Converts a string field to a formatted text field.
   *
   * @param string $entityTypeId
   *   The entity type ID (e.g., 'node', 'user').
   * @param string $fieldName
   *   The name of the field to convert. Best used for text fields.
   *
   * @command druidfire:string2formatted
   *
   * @usage drush druidfire:string2formatted node body
   *   Converts the 'body' field of the 'node' entity type to a formatted text
   *   field.
   */
  public function string2formatted($entityTypeId, $fieldName) {
    // @phpstan-ignore-next-line
    $this->druidfire->string2formatted($entityTypeId, $fieldName);
  }

  /**
   * Converts an Entity Reference Revisions field to an entity reference.
   *
   * @param string $entityTypeId
   *   The entity type ID (e.g., 'node', 'user').
   * @param string $fieldName
   *   The name of the field to convert. Best used for text fields.
   *
   * @command druidfire:err2er
   *
   * @usage drush druidfire:err2er node my_field
   *   Converts the 'my_field' field of the 'node' entity type from ERR to an
   *   entity reference field.
   */
  public function err2er($entityTypeId, $fieldName) {
    // @phpstan-ignore-next-line
    $this->druidfire->err2er($entityTypeId, $fieldName);
  }

  /**
   * Converts an Entity Reference Revisions field to a Bricks field.
   *
   * @param string $entityTypeId
   *   The entity type ID (e.g., 'node', 'user').
   * @param string $fieldName
   *   The name of the field to convert. Best used for paragraph fields.
   *
   * @command druidfire:err2bricks
   *
   * @usage drush druidfire:err2bricks node my_paragraphs_field
   *   Converts the 'my_paragraphs_field' field of the 'node' entity type from
   *   ERR to a Bricks field with depth and options columns.
   */
  public function err2bricks($entityTypeId, $fieldName) {
    // @phpstan-ignore-next-line
    $this->druidfire->err2bricks($entityTypeId, $fieldName);
  }

  /**
   * Converts a string field to a taxonomy term reference field.
   *
   * @param string $entityTypeId
   *   The entity type ID (e.g., 'node', 'user').
   * @param string $fieldName
   *   The name of the field to convert. Best used for string fields.
   * @param string $vid
   *   The vocabulary ID to use for taxonomy terms.
   *
   * @command druidfire:string2taxonomyreference
   *
   * @usage drush druidfire:string2taxonomyreference node category tags
   *   Converts the 'category' field of the 'node' entity type to reference
   *   taxonomy terms from the 'tags' vocabulary.
   */
  public function string2taxonomyreference($entityTypeId, $fieldName, $vid) {
    // @phpstan-ignore-next-line
    $this->druidfire->string2taxonomyReference($entityTypeId, $fieldName, ['vid' => $vid]);
  }

  /**
   * Lists all available Druidfire spells and their purposes.
   *
   * @command druidfire:list-spells
   * @aliases druidfire:ls
   *
   * @usage drush druidfire:list-spells
   *   Shows all available Druidfire spells in a table format.
   */
  public function listSpells() {
    $spells = $this->druidfire->getAvailableSpells();

    // Prepare table headers.
    $headers = ['Command', 'Name', 'Description'];

    // Prepare table rows.
    $rows = [];
    foreach ($spells as $spellId => $definition) {
      $label = (string) $definition['label'];
      $description = (string) $definition['description'];

      $rows[] = [
        "druidfire:$spellId",
        $label,
        $description,
      ];
    }

    // Sort by command name for consistent output.
    usort($rows, fn($rowA, $rowB) => strcmp($rowA[0], $rowB[0]));

    // Create and render table with proper formatting.
    $table = new Table($this->output());
    $table->setHeaders($headers);
    $table->setRows($rows);

    // Set table style for better readability.
    $table->setStyle('box');
    $table->render();

    // Add footer message.
    // @phpstan-ignore-next-line
    $this->io()->note('Use "drush help [command]" for detailed usage information.');
  }

}
