<?php

namespace Drupal\druidfire;

use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Entity\EntityLastInstalledSchemaRepositoryInterface;
use Drupal\Core\Field\FieldTypePluginManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\field\Entity\FieldStorageConfig;

/**
 * Service for managing field and display configuration changes.
 *
 * This service handles the logic for modifying field storage configs,
 * field configs, and form/view display configurations.
 */
class ConfigManager {

  /**
   * Constructs a ConfigManager object.
   *
   * @param \Drupal\Core\Config\StorageInterface $configStorage
   *   The configuration storage service.
   * @param \Drupal\Core\Field\FieldTypePluginManagerInterface $fieldTypePluginManager
   *   The field type plugin manager service.
   * @param \Drupal\Core\Entity\EntityLastInstalledSchemaRepositoryInterface $entityLastInstalledSchemaRepository
   *   The entity last installed schema repository service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    protected StorageInterface $configStorage,
    protected FieldTypePluginManagerInterface $fieldTypePluginManager,
    protected EntityLastInstalledSchemaRepositoryInterface $entityLastInstalledSchemaRepository,
    protected MessengerInterface $messenger,
  ) {}

  /**
   * Changes the configuration for a field.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param string $fieldName
   *   The field name.
   * @param \Drupal\druidfire\SpellInterface $spell
   *   The spell instance.
   * @param string $configType
   *   The configuration type ('storage' or 'field').
   * @param array $optionalArguments
   *   Additional arguments for the spell.
   */
  public function changeConfig(string $entityTypeId, string $fieldName, SpellInterface $spell, string $configType, array $optionalArguments): void {
    $callable = [$spell, $configType];
    $configNames = $this->configStorage->listAll("field.$configType.$entityTypeId.");
    $configNames = preg_grep("/\\.$fieldName$/", $configNames);
    $this->doChangeConfig($callable, $configNames, [$optionalArguments]);

    if ($configType === 'field') {
      array_unshift($optionalArguments, $fieldName);
      $callable[1] = 'formDisplay';
      $this->doChangeConfig($callable, $configNames, [$fieldName, $optionalArguments], "core.entity_form_display.$entityTypeId.");
      $callable[1] = 'viewDisplay';
      $this->doChangeConfig($callable, $configNames, [$fieldName, $optionalArguments], "core.entity_view_display.$entityTypeId.");
    }
  }

  /**
   * Applies changes to configuration records.
   *
   * @param callable $callable
   *   The callable to apply to the configuration.
   * @param array $configNames
   *   The list of configuration names.
   * @param array $args
   *   The arguments to pass to the callable.
   * @param string $configPrefix
   *   The configuration prefix (optional).
   */
  protected function doChangeConfig($callable, array $configNames, array $args, string $configPrefix = ''): void {
    foreach ($configNames as $configName) {
      if ($configPrefix !== '') {
        $actualConfigNames = $this->configStorage->listAll($configPrefix . explode('.', $configName)[3] . '.');
      }
      else {
        $actualConfigNames = [$configName];
      }

      foreach ($actualConfigNames as $actualConfigName) {
        if ($record = $this->configStorage->read($actualConfigName)) {
          $this->messenger->addMessage("Changing config $actualConfigName");
          $record = $callable($record, ...$args);
          $this->configStorage->write($actualConfigName, $record);

          if (str_starts_with($actualConfigName, 'field.storage.')) {
            $class = $this->fieldTypePluginManager->getPluginClass($record['type']);
            $record['settings'] = $class::storageSettingsFromConfigData($record['settings']);
            $this->entityLastInstalledSchemaRepository->setLastInstalledFieldStorageDefinition(new FieldStorageConfig($record));
          }
        }
      }
    }
  }

}
