<?php

namespace Drupal\druidfire;

use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreInterface;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * The main Druidfire class.
 *
 * This class provides functionality to dynamically modify entity field schemas
 * and configurations in a Drupal site. It allows for schema changes, config
 * updates, and caching of entity definitions.
 */
class Druidfire {

  /**
   * Key-value store for entity storage schema data.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueStoreInterface
   */
  protected KeyValueStoreInterface $keyvalue;

  /**
   * The spell plugin manager.
   *
   * @var \Drupal\druidfire\SpellManager
   */
  protected SpellManager $spellManager;

  /**
   * Constructs a Druidfire object.
   *
   * @param \Drupal\Core\KeyValueStore\KeyValueFactoryInterface $keyValueFactory
   *   The key-value factory service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\druidfire\FieldInspector $fieldInspector
   *   The field inspector service.
   * @param \Drupal\druidfire\ConfigManager $configManager
   *   The configuration manager service.
   * @param \Drupal\druidfire\SpellManager $spellManager
   *   The spell plugin manager service.
   */
  public function __construct(
    protected KeyValueFactoryInterface $keyValueFactory,
    protected MessengerInterface $messenger,
    protected FieldInspector $fieldInspector,
    protected ConfigManager $configManager,
    SpellManager $spellManager,
  ) {
    $this->keyvalue = $keyValueFactory->get('entity.storage_schema.sql');
    $this->spellManager = $spellManager;
  }

  /**
   * Magic method to handle dynamic schema changes.
   *
   * @param string $spellName
   *   The name of the spell to apply.
   * @param array $arguments
   *   The arguments for the spell.
   */
  public function __call(string $spellName, array $arguments) {
    $entityTypeId = array_shift($arguments);
    $fieldName = array_shift($arguments);
    $optionalArguments = array_shift($arguments) ?? [];
    $columnName = $this->fieldInspector->getColumnName($entityTypeId, $fieldName, $optionalArguments['property'] ?? NULL);
    $key = "$entityTypeId.field_schema_data.$fieldName";

    // Create spell instance using the plugin manager.
    $spell = $this->spellManager->createSpell($spellName);
    $schema = $this->keyvalue->get($key, []);

    foreach (array_keys($schema) as $tableName) {
      try {
        $this->messenger->addMessage("Changing the schema of $tableName $columnName");
        $schema = $spell->schema($schema, $tableName, $columnName, $optionalArguments);
      }
      catch (\Exception $e) {
        $this->messenger->addWarning("Schema change failed " . $e->getMessage());
      }
    }
    $this->keyvalue->set($key, $schema);
    $this->configManager->changeConfig($entityTypeId, $fieldName, $spell, 'storage', $optionalArguments);
    $this->configManager->changeConfig($entityTypeId, $fieldName, $spell, 'field', $optionalArguments);
    $this->fieldInspector->clearCachedDefinitions();
  }

  /**
   * Gets all available spells from the plugin manager.
   *
   * @return array
   *   An array of spell plugin definitions, keyed by plugin ID.
   */
  public function getAvailableSpells(): array {
    return $this->spellManager->getAvailableSpells();
  }

}
