<?php

namespace Drupal\druidfire;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Sql\DefaultTableMapping;
use Drupal\Core\Entity\Sql\SqlContentEntityStorage;
use Drupal\Core\Entity\Sql\SqlEntityStorageInterface;

/**
 * Service for inspecting entity fields and their database representations.
 *
 * This service handles the logic for resolving field definitions,
 * column names, and table mappings for entity fields.
 */
class FieldInspector {

  /**
   * Constructs a FieldInspector object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager service.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityFieldManagerInterface $entityFieldManager,
  ) {}

  /**
   * Gets the column name for a field.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param string $fieldName
   *   The field name.
   * @param string|null $propertyName
   *   The property name (optional).
   *
   * @return string
   *   The column name.
   *
   * @throws \InvalidArgumentException
   *   Thrown if the field is not a default SQL field.
   */
  public function getColumnName(string $entityTypeId, string $fieldName, ?string $propertyName): string {
    $entityStorage = $this->entityTypeManager->getStorage($entityTypeId);
    assert($entityStorage instanceof SqlEntityStorageInterface);
    $mapping = $entityStorage->getTableMapping();
    $definitions = $this->entityFieldManager->getFieldStorageDefinitions($entityTypeId);

    if (!isset($definitions[$fieldName]) || !$mapping instanceof DefaultTableMapping || !$entityStorage instanceof SqlContentEntityStorage) {
      throw new \InvalidArgumentException('Can only change default SQL fields.');
    }

    $definition = $definitions[$fieldName];
    if ($propertyName !== NULL && $propertyName !== '') {
      $property = $propertyName;
    }
    else {
      $property = $definition->getMainPropertyName();
    }

    return $mapping->getFieldColumnName($definition, $property);
  }

  /**
   * Clears cached entity definitions.
   */
  public function clearCachedDefinitions(): void {
    $this->entityTypeManager->clearCachedDefinitions();
  }

}
