<?php

namespace Drupal\druidfire\Plugin\Spell;

use Drupal\druidfire\SpellBase;

/**
 * Converts an entity reference revisions field to a bricks field.
 *
 * This class modifies the schema, storage, and configuration of an entity
 * reference revisions field to transform it into a bricks field with
 * additional properties.
 *
 * @Spell(
 *   id = "err2bricks",
 *   label = @Translation("ERR to Bricks"),
 *   description = @Translation("Converts Entity Reference Revisions fields to Bricks fields."),
 * )
 */
class Err2Bricks extends SpellBase {

  /**
   * Modifies the schema of a field table.
   *
   * Adds new columns for bricks-specific properties, such as depth and options.
   *
   * @param array $schema
   *   The current table schema.
   * @param string $tableName
   *   The name of the field table (e.g., node__field_bricks).
   * @param string $columnName
   *   The column of the field property being changed
   *   (e.g., field_bricks_target_id).
   * @param array $args
   *   Additional arguments required for the schema modification.
   *
   * @return array
   *   The modified table schema.
   */
  public function schema(array $schema, string $tableName, string $columnName, array $args = []): array {
    $basename = str_replace('_target_id', '', $columnName);
    $new_columns = [
      "{$basename}_depth" => [
        "type" => "int",
        "size" => "tiny",
        "unsigned" => TRUE,
        "not null" => FALSE,
      ],
      "{$basename}_options" => [
        "type" => "blob",
        "size" => "normal",
        "not null" => FALSE,
        "serialize" => TRUE,
      ],
    ];
    foreach ($new_columns as $new_column_name => $new_column) {
      $this->schema->addField($tableName, $new_column_name, $new_column);
    }
    $schema[$tableName]['fields'] = $schema[$tableName]['fields'] + $new_columns;
    return $schema;
  }

  /**
   * Modifies the field storage configuration.
   *
   * Changes the field type to 'bricks_revisioned'.
   *
   * @param array $yaml
   *   The field storage configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified field storage configuration array.
   */
  public function storage(array $yaml, array $args = []): array {
    $yaml['type'] = 'bricks_revisioned';
    return $yaml;
  }

  /**
   * Modifies the field configuration for a specific bundle.
   *
   * Configures the field to use the 'bricks_revisioned' field type.
   *
   * @param array $yaml
   *   The field configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified field configuration array.
   */
  public function field(array $yaml, array $args = []): array {
    $yaml['field_type'] = 'bricks_revisioned';
    return $yaml;
  }

  /**
   * Modifies the form display configuration for a field.
   *
   * Leaves the form display configuration unchanged, as bricks fields can
   * use the standard paragraphs widget or other widgets configured in the UI.
   *
   * @param array $yaml
   *   The form display configuration array.
   * @param string $fieldName
   *   The name of the field.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified form display configuration array.
   */
  public function formDisplay(array $yaml, $fieldName, array $args = []): array {
    return $yaml;
  }

  /**
   * Modifies the view display configuration for a field.
   *
   * Sets the widget type to 'bricks_revisions_nested' for view displays.
   *
   * @param array $yaml
   *   The view display configuration array.
   * @param string $fieldName
   *   The name of the field.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified view display configuration array.
   */
  public function viewDisplay(array $yaml, $fieldName, array $args = []): array {
    if (isset($yaml['content'][$fieldName]['type'])) {
      $yaml['content'][$fieldName]['type'] = 'bricks_revisions_nested';
    }
    return $yaml;
  }

}
