<?php

namespace Drupal\druidfire\Plugin\Spell;

use Drupal\druidfire\SpellBase;

/**
 * Resizes a text or string field.
 *
 * This class modifies the schema and storage configuration of a text or string
 * field to adjust its maximum length.
 *
 * @Spell(
 *   id = "resize",
 *   label = @Translation("Resize"),
 *   description = @Translation("Resizes a text or string field to a new length."),
 * )
 */
class Resize extends SpellBase {

  /**
   * Default size for the field length if no size is provided.
   */
  const DEFAULTSIZE = 1024;

  /**
   * Modifies the schema of a field table.
   *
   * Updates the length of the specified column in the schema to the provided
   * size or the default size if none is specified.
   *
   * @param array $schema
   *   The current table schema.
   * @param string $tableName
   *   The name of the field table (e.g., node__field_name).
   * @param string $columnName
   *   The column of the field property being changed (e.g., field_name_value).
   * @param array $args
   *   Additional arguments required for the schema modification.
   *   - 'size': (optional) The new size for the field.
   *
   * @return array
   *   The modified table schema.
   */
  public function schema(array $schema, string $tableName, string $columnName, array $args = []): array {
    $schema[$tableName]['fields'][$columnName]['length'] = $args['size'] ?? self::DEFAULTSIZE;
    $this->schema->changeField($tableName, $columnName, $columnName, $schema[$tableName]['fields'][$columnName]);
    return $schema;
  }

  /**
   * Modifies the field storage configuration.
   *
   * Updates the maximum length of the field in the storage configuration to
   * the provided size or the default size if none is specified.
   *
   * @param array $yaml
   *   The field storage configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *   - 'size': (optional) The new size for the field.
   *
   * @return array
   *   The modified field storage configuration array.
   */
  public function storage(array $yaml, array $args = []): array {
    $yaml['settings']['max_length'] = $args['size'] ?? self::DEFAULTSIZE;
    return $yaml;
  }

}
