<?php

namespace Drupal\druidfire;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Schema;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for implementing spells in the Druidfire module.
 *
 * This class provides default implementations for the SpellInterface methods.
 * It can be extended by specific spell classes to modify entity field schemas,
 * storage, and configurations dynamically.
 */
class SpellBase extends PluginBase implements SpellInterface, ContainerFactoryPluginInterface {

  /**
   * The database connection service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The database schema service.
   *
   * @var \Drupal\Core\Database\Schema
   */
  protected Schema $schema;

  /**
   * Constructs a SpellBase object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, Connection $database) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->database = $database;
    $this->schema = $this->database->schema();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database')
    );
  }

  /**
   * Modifies the schema of a field table.
   *
   * This is a default implementation that returns the schema unchanged.
   *
   * @param array $schema
   *   The current table schema.
   * @param string $tableName
   *   The name of the field table (e.g., node__body).
   * @param string $columnName
   *   The column of the field property being changed (e.g., body_value).
   * @param array $args
   *   Additional arguments required for the schema modification.
   *
   * @return array
   *   The modified table schema.
   */
  public function schema(array $schema, string $tableName, string $columnName, array $args = []): array {
    return $schema;
  }

  /**
   * Modifies the field storage configuration.
   *
   * This is a default implementation that returns the configuration unchanged.
   *
   * @param array $yaml
   *   The field storage configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified field storage configuration array.
   */
  public function storage(array $yaml, array $args = []): array {
    return $yaml;
  }

  /**
   * Modifies the field configuration for a specific bundle.
   *
   * This is a default implementation that returns the configuration unchanged.
   *
   * @param array $yaml
   *   The field configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified field configuration array.
   */
  public function field(array $yaml, array $args = []): array {
    return $yaml;
  }

  /**
   * Modifies the form display configuration for a field.
   *
   * This is a default implementation that returns the configuration unchanged.
   *
   * @param array $yaml
   *   The form display configuration array.
   * @param string $fieldName
   *   The name of the field. Typically, only $yaml['content'][$fieldName]
   *   will need modification.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified form display configuration array.
   */
  public function formDisplay(array $yaml, $fieldName, array $args = []): array {
    return $yaml;
  }

  /**
   * Modifies the view display configuration for a field.
   *
   * This is a default implementation that returns the configuration unchanged.
   *
   * @param array $yaml
   *   The view display configuration array.
   * @param string $fieldName
   *   The name of the field. Typically, only $yaml['content'][$fieldName]
   *   will need modification.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified view display configuration array.
   */
  public function viewDisplay(array $yaml, $fieldName, array $args = []): array {
    return $yaml;
  }

}
