<?php

namespace Drupal\druidfire;

/**
 * Interface for defining spells in the Druidfire module.
 *
 * A spell is a set of operations that can dynamically modify the schema,
 * storage, and configuration of entity fields in a Drupal site. This interface
 * defines the methods required to implement a spell.
 */
interface SpellInterface {

  /**
   * Modifies the schema of a field table.
   *
   * @param array $schema
   *   The current table schema.
   * @param string $tableName
   *   The name of the field table (e.g., node__body).
   * @param string $columnName
   *   The column of the field property being changed (e.g., body_value).
   * @param array $args
   *   Additional arguments required for the schema modification.
   *
   * @return array
   *   The modified table schema.
   */
  public function schema(array $schema, string $tableName, string $columnName, array $args = []): array;

  /**
   * Modifies the field storage configuration.
   *
   * @param array $yaml
   *   The field storage configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified field storage configuration array.
   */
  public function storage(array $yaml, array $args = []): array;

  /**
   * Modifies the field configuration for a specific bundle.
   *
   * @param array $yaml
   *   The field configuration array.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified field configuration array.
   */
  public function field(array $yaml, array $args = []): array;

  /**
   * Modifies the form display configuration for a field.
   *
   * @param array $yaml
   *   The form display configuration array.
   * @param string $fieldName
   *   The name of the field. Typically, only $yaml['content'][$fieldName]
   *   will need modification.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified form display configuration array.
   */
  public function formDisplay(array $yaml, string $fieldName, array $args = []): array;

  /**
   * Modifies the view display configuration for a field.
   *
   * @param array $yaml
   *   The view display configuration array.
   * @param string $fieldName
   *   The name of the field. Typically, only $yaml['content'][$fieldName]
   *   will need modification.
   * @param array $args
   *   Additional arguments required for the modification.
   *
   * @return array
   *   The modified view display configuration array.
   */
  public function viewDisplay(array $yaml, string $fieldName, array $args = []): array;

}
