<?php

namespace Drupal\Tests\druidfire\Unit;

use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\druidfire\ConfigManager;
use Drupal\druidfire\Druidfire;
use Drupal\druidfire\FieldInspector;
use Drupal\druidfire\SpellInterface;
use Drupal\druidfire\SpellManager;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for the Druidfire service.
 *
 * @coversDefaultClass \Drupal\druidfire\Druidfire
 * @group druidfire
 */
class DruidfireTest extends UnitTestCase {

  /**
   * The Druidfire service under test.
   *
   * @var \Drupal\druidfire\Druidfire
   */
  protected Druidfire $druidfire;

  /**
   * Mock key-value factory service.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected KeyValueFactoryInterface|MockObject $keyValueFactory;

  /**
   * Mock messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MessengerInterface|MockObject $messenger;

  /**
   * Mock field inspector service.
   *
   * @var \Drupal\druidfire\FieldInspector|\PHPUnit\Framework\MockObject\MockObject
   */
  protected FieldInspector|MockObject $fieldInspector;

  /**
   * Mock config manager service.
   *
   * @var \Drupal\druidfire\ConfigManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected ConfigManager|MockObject $configManager;

  /**
   * Mock key-value store.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueStoreInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected KeyValueStoreInterface|MockObject $keyValueStore;

  /**
   * Mock spell manager service.
   *
   * @var \Drupal\druidfire\SpellManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected SpellManager|MockObject $spellManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->keyValueFactory = $this->createMock(KeyValueFactoryInterface::class);
    $this->messenger = $this->createMock(MessengerInterface::class);
    $this->fieldInspector = $this->createMock(FieldInspector::class);
    $this->configManager = $this->createMock(ConfigManager::class);
    $this->keyValueStore = $this->createMock(KeyValueStoreInterface::class);
    $this->spellManager = $this->createMock(SpellManager::class);

    $this->keyValueFactory
      ->method('get')
      ->with('entity.storage_schema.sql')
      ->willReturn($this->keyValueStore);

    $this->druidfire = new Druidfire(
      $this->keyValueFactory,
      $this->messenger,
      $this->fieldInspector,
      $this->configManager,
      $this->spellManager
    );
  }

  /**
   * Tests the getAvailableSpells method.
   *
   * @covers ::getAvailableSpells
   */
  public function testGetAvailableSpells(): void {
    $expectedSpells = [
      'resize' => [
        'id' => 'resize',
        'label' => 'Resize',
        'description' => 'Resizes a field',
      ],
      'string2formatted' => [
        'id' => 'string2formatted',
        'label' => 'String to Formatted',
        'description' => 'Converts string to formatted text',
      ],
    ];

    $this->spellManager
      ->expects($this->once())
      ->method('getAvailableSpells')
      ->willReturn($expectedSpells);

    $result = $this->druidfire->getAvailableSpells();

    $this->assertEquals($expectedSpells, $result);
  }

  /**
   * Tests the __call method with successful spell execution.
   *
   * @covers ::__call
   */
  public function testCallMethodSuccess(): void {
    $entityTypeId = 'node';
    $fieldName = 'body';
    $args = ['size' => 255];
    $spellName = 'resize';
    $columnName = 'body_value';

    // Mock spell.
    $spell = $this->createMock(SpellInterface::class);
    $spell->expects($this->once())
      ->method('schema')
      ->willReturn(['test_table' => ['fields' => []]]);

    // Mock spell manager to return the spell.
    $this->spellManager
      ->expects($this->once())
      ->method('createSpell')
      ->with($spellName)
      ->willReturn($spell);

    // Mock field inspector.
    $this->fieldInspector
      ->expects($this->once())
      ->method('getColumnName')
      ->with($entityTypeId, $fieldName, NULL)
      ->willReturn($columnName);

    // Mock key-value store operations.
    $schema = ['test_table' => ['fields' => []]];
    $this->keyValueStore
      ->expects($this->once())
      ->method('get')
      ->with("$entityTypeId.field_schema_data.$fieldName", [])
      ->willReturn($schema);

    $this->keyValueStore
      ->expects($this->once())
      ->method('set')
      ->with("$entityTypeId.field_schema_data.$fieldName", $schema);

    $this->messenger
      ->expects($this->once())
      ->method('addMessage')
      ->with("Changing the schema of test_table $columnName");

    // Mock field inspector clearing cached definitions.
    $this->fieldInspector
      ->expects($this->once())
      ->method('clearCachedDefinitions');

    // Mock config manager operations.
    $this->configManager
      ->expects($this->exactly(2))
      ->method('changeConfig');
    // Note: Due to PHPUnit changes, we can't easily test the exact parameters
    // with withConsecutive, but the integration tests will verify this works.
    // Execute the magic method call.
    // @phpstan-ignore-next-line
    $this->druidfire->$spellName($entityTypeId, $fieldName, $args);
  }

  /**
   * Tests the __call method with exception handling.
   *
   * @covers ::__call
   */
  public function testCallMethodWithException(): void {
    $entityTypeId = 'node';
    $fieldName = 'body';
    $args = ['size' => 255];
    $spellName = 'resize';

    // Mock spell that throws exception.
    $spell = $this->createMock(SpellInterface::class);
    $spell->expects($this->once())
      ->method('schema')
      ->willThrowException(new \Exception('Test exception'));

    // Mock spell manager to return the spell.
    $this->spellManager
      ->expects($this->once())
      ->method('createSpell')
      ->with($spellName)
      ->willReturn($spell);

    // Mock field inspector.
    $this->fieldInspector
      ->method('getColumnName')
      ->willReturn('body_value');

    $this->keyValueStore
      ->method('get')
      ->willReturn(['test_table' => ['fields' => []]]);

    $this->messenger
      ->expects($this->once())
      ->method('addWarning')
      ->with('Schema change failed Test exception');

    // Execute the magic method call.
    // @phpstan-ignore-next-line
    $this->druidfire->$spellName($entityTypeId, $fieldName, $args);
  }

  /**
   * Tests field inspector exception propagation.
   */
  public function testFieldInspectorException(): void {
    $entityTypeId = 'node';
    $fieldName = 'nonexistent_field';
    $args = [];
    $spellName = 'resize';

    // Mock field inspector to throw exception before spell manager is called.
    $this->fieldInspector
      ->method('getColumnName')
      ->willThrowException(new \InvalidArgumentException('Can only change default SQL fields.'));

    // Spell manager should not be called when field inspector throws exception.
    $this->spellManager
      ->expects($this->never())
      ->method('createSpell');

    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Can only change default SQL fields.');

    // Execute the magic method call.
    // @phpstan-ignore-next-line
    $this->druidfire->$spellName($entityTypeId, $fieldName, $args);
  }

}
