<?php

namespace Drupal\Tests\druidfire\Unit\Plugin\Spell;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Schema;
use Drupal\druidfire\Plugin\Spell\Resize;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for the Resize spell plugin.
 *
 * @coversDefaultClass \Drupal\druidfire\Plugin\Spell\Resize
 * @group druidfire
 */
class ResizeTest extends UnitTestCase {

  /**
   * The Resize spell plugin under test.
   *
   * @var \Drupal\druidfire\Plugin\Spell\Resize
   */
  protected Resize $resize;

  /**
   * Mock database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Connection|MockObject $database;

  /**
   * Mock database schema.
   *
   * @var \Drupal\Core\Database\Schema|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Schema|MockObject $schema;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->database = $this->createMock(Connection::class);
    $this->schema = $this->createMock(Schema::class);

    $this->database
      ->method('schema')
      ->willReturn($this->schema);

    $this->resize = new Resize([], 'resize', [], $this->database);
  }

  /**
   * Tests schema modification with custom size.
   *
   * @covers ::schema
   */
  public function testSchemaWithCustomSize(): void {
    $schema = [
      'node__body' => [
        'fields' => [
          'body_value' => [
            'type' => 'varchar',
            'length' => 255,
            'not null' => FALSE,
          ],
        ],
      ],
    ];

    $tableName = 'node__body';
    $columnName = 'body_value';
    $args = ['size' => 512];

    $this->schema
      ->expects($this->once())
      ->method('changeField')
      ->with(
        $tableName,
        $columnName,
        $columnName,
        ['type' => 'varchar', 'length' => 512, 'not null' => FALSE]
      );

    $result = $this->resize->schema($schema, $tableName, $columnName, $args);

    $expected = [
      'node__body' => [
        'fields' => [
          'body_value' => [
            'type' => 'varchar',
            'length' => 512,
            'not null' => FALSE,
          ],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests storage configuration with custom size.
   *
   * @covers ::storage
   */
  public function testStorageWithCustomSize(): void {
    $yaml = [
      'type' => 'string',
      'settings' => [
        'max_length' => 255,
        'case_sensitive' => FALSE,
      ],
    ];

    $args = ['size' => 1024];

    $result = $this->resize->storage($yaml, $args);

    $expected = [
      'type' => 'string',
      'settings' => [
        'max_length' => 1024,
        'case_sensitive' => FALSE,
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests default size constant.
   */
  public function testDefaultSizeConstant(): void {
    $this->assertEquals(1024, Resize::DEFAULTSIZE);
  }

}
