<?php

namespace Drupal\Tests\druidfire\Unit\Plugin\Spell;

use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Database\Query\Update;
use Drupal\Core\Database\Schema;
use Drupal\druidfire\Plugin\Spell\String2TaxonomyReference;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for the String2TaxonomyReference spell plugin.
 *
 * @coversDefaultClass \Drupal\druidfire\Plugin\Spell\String2TaxonomyReference
 * @group druidfire
 */
class String2TaxonomyReferenceTest extends UnitTestCase {

  /**
   * The String2TaxonomyReference spell plugin under test.
   *
   * @var \Drupal\druidfire\Plugin\Spell\String2TaxonomyReference
   */
  protected String2TaxonomyReference $string2TaxonomyReference;

  /**
   * Mock database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Connection|MockObject $database;

  /**
   * Mock database schema.
   *
   * @var \Drupal\Core\Database\Schema|\PHPUnit\Framework\MockObject\MockObject
   */
  protected Schema|MockObject $schema;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->database = $this->createMock(Connection::class);
    $this->schema = $this->createMock(Schema::class);

    $this->database
      ->method('schema')
      ->willReturn($this->schema);

    $this->string2TaxonomyReference = new String2TaxonomyReference([], 'string2taxonomyReference', [], $this->database);
  }

  /**
   * Tests schema conversion throws exception without vid.
   *
   * @covers ::schema
   */
  public function testSchemaWithoutVidThrowsException(): void {
    $schema = [
      'node__field_category' => [
        'fields' => [
          'field_category_value' => [
            'type' => 'varchar',
            'length' => 255,
          ],
        ],
      ],
    ];

    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('vid argument is required for string2taxonomyReference schema conversion.');

    $this->string2TaxonomyReference->schema($schema, 'node__field_category', 'field_category_value', []);
  }

  /**
   * Tests schema converts string to taxonomy reference.
   *
   * @covers ::schema
   */
  public function testSchemaConvertsStringToTaxonomyReference(): void {
    $schema = [
      'node__field_category' => [
        'fields' => [
          'field_category_value' => [
            'type' => 'varchar',
            'length' => 255,
            'not null' => FALSE,
          ],
        ],
      ],
    ];

    $tableName = 'node__field_category';
    $columnName = 'field_category_value';
    $args = ['vid' => 'tags'];

    // Mock database queries for term creation and population.
    $subquery = $this->createMock(SelectInterface::class);
    $query = $this->createMock(SelectInterface::class);
    $update = $this->createMock(Update::class);

    $this->database
      ->expects($this->exactly(2))
      ->method('select')
      ->willReturnOnConsecutiveCalls($subquery, $query);

    $this->database
      ->method('update')
      ->willReturn($update);

    $this->database
      ->method('escapeField')
      ->with($columnName)
      ->willReturn('field_category_value');

    // Configure subquery mock.
    $subquery->method('fields')->willReturnSelf();
    $subquery->method('where')->willReturnSelf();

    // Configure main query mock.
    $query->method('distinct')->willReturnSelf();
    $query->method('addField')->willReturnSelf();
    $query->method('addExpression')->willReturnSelf();
    $query->method('notExists')->willReturnSelf();

    // Create a mock result that has fetchAll method.
    $result = $this->createMock(StatementInterface::class);
    $result->method('fetchAll')->willReturn([]);
    $query->method('execute')->willReturn($result);

    // Configure update query mock.
    $update->method('expression')->willReturnSelf();
    $update->method('execute')->willReturn(1);

    // Expect new target_id column to be added.
    $this->schema
      ->expects($this->once())
      ->method('addField')
      ->with(
        $tableName,
        'field_category_target_id',
        [
          'description' => 'The ID of the target entity.',
          'type' => 'int',
          'unsigned' => TRUE,
        ]
      );

    // Expect original column to be dropped.
    $this->schema
      ->expects($this->once())
      ->method('dropField')
      ->with($tableName, $columnName);

    $result = $this->string2TaxonomyReference->schema($schema, $tableName, $columnName, $args);

    $expected = [
      'node__field_category' => [
        'fields' => [
          'field_category_value' => [
            'type' => 'varchar',
            'length' => 255,
            'not null' => FALSE,
          ],
          'field_category_target_id' => [
            'description' => 'The ID of the target entity.',
            'type' => 'int',
            'unsigned' => TRUE,
          ],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests storage configuration changes to entity_reference.
   *
   * @covers ::storage
   */
  public function testStorageChangesToEntityReference(): void {
    $yaml = [
      'type' => 'string',
      'settings' => [
        'max_length' => 255,
      ],
    ];

    $args = [];

    $result = $this->string2TaxonomyReference->storage($yaml, $args);

    $expected = [
      'type' => 'entity_reference',
      'settings' => [
        'target_type' => 'taxonomy_term',
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests field configuration for taxonomy reference.
   *
   * @covers ::field
   */
  public function testFieldConfiguresTaxonomyReference(): void {
    $yaml = [
      'field_type' => 'string',
      'settings' => [
        'max_length' => 255,
      ],
    ];

    $args = ['vid' => 'tags'];

    $result = $this->string2TaxonomyReference->field($yaml, $args);

    $expected = [
      'field_type' => 'entity_reference',
      'settings' => [
        'handler' => 'default:taxonomy_term',
        'handler_settings' => [
          'target_bundles' => ['tags' => 'tags'],
          'auto_create' => FALSE,
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests field configuration without vid.
   *
   * @covers ::field
   */
  public function testFieldWithoutVid(): void {
    $yaml = [
      'field_type' => 'string',
      'settings' => [
        'max_length' => 255,
      ],
    ];

    $args = [];

    $result = $this->string2TaxonomyReference->field($yaml, $args);

    $expected = [
      'field_type' => 'entity_reference',
      'settings' => [
        'handler' => 'default:taxonomy_term',
        'handler_settings' => [
          'target_bundles' => [],
          'auto_create' => FALSE,
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests form display updates widget to entity_reference_label.
   *
   * @covers ::formDisplay
   */
  public function testFormDisplayUpdatesWidget(): void {
    $yaml = [
      'content' => [
        'field_category' => [
          'type' => 'string_textfield',
          'settings' => [
            'size' => 60,
          ],
        ],
      ],
    ];

    $fieldName = 'field_category';
    $args = [];

    $result = $this->string2TaxonomyReference->formDisplay($yaml, $fieldName, $args);

    $expected = [
      'content' => [
        'field_category' => [
          'type' => 'entity_reference_label',
          'settings' => [
            'link' => FALSE,
          ],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests view display updates widget to options_select.
   *
   * @covers ::viewDisplay
   */
  public function testViewDisplayUpdatesWidget(): void {
    $yaml = [
      'content' => [
        'field_category' => [
          'type' => 'string',
          'settings' => [],
        ],
      ],
    ];

    $fieldName = 'field_category';
    $args = [];

    $result = $this->string2TaxonomyReference->viewDisplay($yaml, $fieldName, $args);

    $expected = [
      'content' => [
        'field_category' => [
          'type' => 'options_select',
          'settings' => [],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests form display with missing field adds the field.
   *
   * @covers ::formDisplay
   */
  public function testFormDisplayWithMissingField(): void {
    $yaml = [
      'content' => [
        'other_field' => [
          'type' => 'string_textfield',
        ],
      ],
    ];

    $fieldName = 'field_category';
    $args = [];

    $result = $this->string2TaxonomyReference->formDisplay($yaml, $fieldName, $args);

    $expected = [
      'content' => [
        'other_field' => [
          'type' => 'string_textfield',
        ],
        'field_category' => [
          'type' => 'entity_reference_label',
          'settings' => [
            'link' => FALSE,
          ],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests view display with missing field adds the field.
   *
   * @covers ::viewDisplay
   */
  public function testViewDisplayWithMissingField(): void {
    $yaml = [
      'content' => [
        'other_field' => [
          'type' => 'string',
        ],
      ],
    ];

    $fieldName = 'field_category';
    $args = [];

    $result = $this->string2TaxonomyReference->viewDisplay($yaml, $fieldName, $args);

    $expected = [
      'content' => [
        'other_field' => [
          'type' => 'string',
        ],
        'field_category' => [
          'type' => 'options_select',
          'settings' => [],
        ],
      ],
    ];

    $this->assertEquals($expected, $result);
  }

}
