<?php

declare(strict_types=1);

namespace Drupal\Tests\migrate_drupal\Kernel;

use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\migrate_drupal\Plugin\MigrateFieldPluginManager;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\IgnoreDeprecations;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the field plugin manager.
 */
#[CoversClass(MigrateFieldPluginManager::class)]
#[Group('migrate_drupal')]
#[IgnoreDeprecations]
#[RunTestsInSeparateProcesses]
class MigrateFieldPluginManagerTest extends MigrateDrupalTestBase {

  /**
   * The field plugin manager.
   *
   * @var \Drupal\migrate_drupal\Plugin\MigrateFieldPluginManagerInterface
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'datetime',
    'field',
    'migrate_drupal',
    'options',
    'file',
    'image',
    'text',
    'link',
    'migrate_field_plugin_manager_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->pluginManager = $this->container->get('plugin.manager.migrate.field');
  }

  /**
   * Tests that the correct MigrateField plugins are used.
   *
   * @legacy-covers ::getPluginIdFromFieldType
   */
  public function testPluginSelection(): void {
    $this->assertSame('link', $this->pluginManager->getPluginIdFromFieldType('link', ['core' => 6]));
    $this->assertSame('link_field', $this->pluginManager->getPluginIdFromFieldType('link_field', ['core' => 7]));
    $this->assertSame('image', $this->pluginManager->getPluginIdFromFieldType('image', ['core' => 7]));
    $this->assertSame('file', $this->pluginManager->getPluginIdFromFieldType('file', ['core' => 7]));
    $this->assertSame('d6_file', $this->pluginManager->getPluginIdFromFieldType('file', ['core' => 6]));
    $this->assertSame('d6_text', $this->pluginManager->getPluginIdFromFieldType('text', ['core' => 6]));
    $this->assertSame('d7_text', $this->pluginManager->getPluginIdFromFieldType('text', ['core' => 7]));

    // Test that the deprecated d6 'date' plugin is not returned.
    $this->assertSame('datetime', $this->pluginManager->getPluginIdFromFieldType('date', ['core' => 6]));

    // Test fallback when no core version is specified.
    $this->assertSame('d6_no_core_version_specified', $this->pluginManager->getPluginIdFromFieldType('d6_no_core_version_specified', ['core' => 6]));
  }

  /**
   * Tests that a PluginNotFoundException is thrown when a plugin isn't found.
   *
   * @legacy-covers ::getPluginIdFromFieldType
   */
  #[DataProvider('nonExistentPluginExceptionsData')]
  public function testNonExistentPluginExceptions($core, $field_type): void {
    $this->expectException(PluginNotFoundException::class);
    $this->expectExceptionMessage(sprintf("Plugin ID '%s' was not found.", $field_type));
    $this->pluginManager->getPluginIdFromFieldType($field_type, ['core' => $core]);
  }

  /**
   * Provides data for testNonExistentPluginExceptions.
   *
   * @return array
   *   The data.
   */
  public static function nonExistentPluginExceptionsData() {
    return [
      'D7 Filefield' => [
        'core' => 7,
        'field_type' => 'filefield',
      ],
      'D6 linkfield' => [
        'core' => 6,
        'field_type' => 'link_field',
      ],
      'D7 link' => [
        'core' => 7,
        'field_type' => 'link',
      ],
      'D7 no core version' => [
        'core' => 7,
        'field_type' => 'd6_no_core_version_specified',
      ],
    ];
  }

  /**
   * Tests that plugins with no explicit weight are given a weight of 0.
   */
  public function testDefaultWeight(): void {
    $definitions = $this->pluginManager->getDefinitions();
    $deprecated_plugins = [
      'date',
    ];
    foreach ($definitions as $id => $definition) {
      $this->assertArrayHasKey('weight', $definition);
      if (in_array($id, $deprecated_plugins, TRUE)) {
        $this->assertSame(9999999, $definition['weight']);
      }
      else {
        $this->assertSame(0, $definition['weight']);
      }
    }
  }

}
