<?php

declare(strict_types=1);

namespace Drupal\KernelTests\Core\Entity;

use Drupal\Core\TypedData\DataDefinition;
use Drupal\KernelTests\KernelTestBase;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests validation constraints for BundleConstraintValidator.
 */
#[Group('Entity')]
#[RunTestsInSeparateProcesses]
class BundleConstraintValidatorTest extends KernelTestBase {

  /**
   * The typed data manager to use.
   *
   * @var \Drupal\Core\TypedData\TypedDataManager
   */
  protected $typedDataManager;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['node', 'field', 'text', 'user'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->typedDataManager = $this->container->get('typed_data_manager');
  }

  /**
   * Tests bundle constraint validation.
   */
  public function testValidation(): void {
    // Test with multiple values.
    $this->assertValidation(['foo', 'bar']);
    // Test with a single string value as well.
    $this->assertValidation('foo');
  }

  /**
   * Executes the BundleConstraintValidator test for a given bundle.
   *
   * @param string|array $bundle
   *   Bundle/bundles to use as constraint option.
   *
   * @internal
   */
  protected function assertValidation($bundle): void {
    // Create a typed data definition with a Bundle constraint.
    $definition = DataDefinition::create('entity_reference')
      ->addConstraint('Bundle', $bundle);

    // Test the validation.
    $node = $this->container->get('entity_type.manager')->getStorage('node')->create(['type' => 'foo']);

    $typed_data = $this->typedDataManager->create($definition, $node);
    $violations = $typed_data->validate();
    $this->assertEquals(0, $violations->count(), 'Validation passed for correct value.');

    // Test the validation when an invalid value is passed.
    $page_node = $this->container->get('entity_type.manager')->getStorage('node')->create(['type' => 'baz']);

    $typed_data = $this->typedDataManager->create($definition, $page_node);
    $violations = $typed_data->validate();
    $this->assertEquals(1, $violations->count(), 'Validation failed for incorrect value.');

    // Make sure the information provided by a violation is correct.
    $violation = $violations[0];
    $this->assertEquals(sprintf('The entity must be of bundle %s.', implode(', ', (array) $bundle)), $violation->getMessage(), 'The message for invalid value is correct.');
    $this->assertEquals($typed_data, $violation->getRoot(), 'Violation root is correct.');
    $this->assertEquals($page_node, $violation->getInvalidValue(), 'The invalid value is set correctly in the violation.');
  }

}
