<?php

declare(strict_types=1);

namespace Drupal\Tests\contact\Kernel;

use Drupal\contact\Entity\ContactForm;
use Drupal\Core\Config\Action\ConfigActionManager;
use Drupal\KernelTests\KernelTestBase;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests Config Actions.
 */
#[Group('contact')]
#[RunTestsInSeparateProcesses]
class ConfigActionsTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['contact', 'system', 'user'];

  /**
   * The configuration action manager.
   *
   * @var \Drupal\Core\Config\Action\ConfigActionManager
   */
  private readonly ConfigActionManager $configActionManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig('contact');
    $this->configActionManager = $this->container->get('plugin.manager.config_action');
  }

  /**
   * Tests the application of configuration actions on a contact form.
   */
  public function testConfigActions(): void {
    $form = ContactForm::load('personal');
    $this->assertSame('Your message has been sent.', $form->getMessage());
    $this->assertEmpty($form->getRecipients());
    $this->assertSame('/', $form->getRedirectUrl()->toString());
    $this->assertEmpty($form->getReply());
    $this->assertSame(0, $form->getWeight());

    $this->configActionManager->applyAction(
      'entity_method:contact.form:setMessage',
      $form->getConfigDependencyName(),
      'Fly, little message!',
    );
    $this->configActionManager->applyAction(
      'entity_method:contact.form:setRecipients',
      $form->getConfigDependencyName(),
      ['ben@deep.space', 'jake@deep.space'],
    );
    $this->configActionManager->applyAction(
      'entity_method:contact.form:setRedirectPath',
      $form->getConfigDependencyName(),
      '/admin/appearance',
    );
    $this->configActionManager->applyAction(
      'entity_method:contact.form:setReply',
      $form->getConfigDependencyName(),
      "From hell's heart, I reply to thee.",
    );
    $this->configActionManager->applyAction(
      'entity_method:contact.form:setWeight',
      $form->getConfigDependencyName(),
      -10,
    );

    $form = ContactForm::load($form->id());
    $this->assertSame('Fly, little message!', $form->getMessage());
    $this->assertSame(['ben@deep.space', 'jake@deep.space'], $form->getRecipients());
    $this->assertSame('/admin/appearance', $form->getRedirectUrl()->toString());
    $this->assertSame("From hell's heart, I reply to thee.", $form->getReply());
    $this->assertSame(-10, $form->getWeight());
  }

}
