<?php

declare(strict_types=1);

namespace Drupal\Tests\Core\Mail\Plugin\Mail;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Mail\MailFormatHelper;
use Drupal\Core\Mail\Plugin\Mail\SymfonyMailer;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use Psr\Log\LoggerInterface;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Email;

/**
 * Tests Drupal\Core\Mail\Plugin\Mail\SymfonyMailer.
 */
#[CoversClass(SymfonyMailer::class)]
#[Group('Mail')]
class SymfonyMailerTest extends UnitTestCase {

  /**
   * Tests that mixed plain text and html body is converted correctly.
   */
  public function testFormatResemblesHtml(): void {
    // Populate global $base_path to avoid notices generated by
    // MailFormatHelper::htmlToMailUrls()
    global $base_path;
    $original_base_path = $base_path;
    $base_path = '/';

    $variables = [
      '@form-url' => 'https://www.example.com/contact',
      '@sender-url' => 'https://www.example.com/user/123',
      '@sender-name' => $this->randomString(),
    ];

    $plain = "In HTML, ampersand must be written as &amp;.\nI saw your house and <wow> it is great. There is too much to say about that beautiful building, it will never fit on one line of text.\nIf a<b and b<c then a<c.";
    $template = "@sender-name (@sender-url) sent a message using the contact form at @form-url.";
    $markup = new FormattableMarkup($template, $variables);

    $message = [
      'body' => [
        $plain,
        $markup,
      ],
    ];

    /** @var \Symfony\Component\Mailer\MailerInterface|\PHPUnit\Framework\MockObject\MockObject */
    $mailer = $this->getMockBuilder(MailerInterface::class)->getMock();

    /** @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject */
    $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

    $plugin = new SymfonyMailer($logger, $mailer);
    $message = $plugin->format($message);

    $expect = MailFormatHelper::wrapMail($plain . "\n\n" . strtr($template, $variables) . "\n");
    $this->assertEquals($expect, $message['body']);

    $base_path = $original_base_path;
  }

  /**
   * Tests sending a mail with special characters in various fields.
   *
   * @param array $message
   *   An array containing the following keys for an email message:
   *     - from
   *     - reply-to
   *     - to
   *     - subject.
   * @param array $expected
   *   An array containing the following keys from the email headers from the
   *   sent email:
   *     - from
   *     - reply-to
   *     - to
   *     - subject.
   *
   * @legacy-covers ::mail
   */
  #[DataProvider('providerMail')]
  public function testMail(array $message, array $expected): void {
    // Setup a mail message.
    $message += [
      'id' => 'example_key',
      'module' => 'example',
      'key' => 'key',
      'langcode' => 'en',
      'params' => [],
      'send' => TRUE,
      'body' => '',
      'headers' => [
        'MIME-Version' => '1.0',
        'Content-Type' => 'text/plain; charset=utf-8; format=flowed; delsp=yes',
        'Content-Transfer-Encoding' => '8Bit',
        'X-Mailer' => 'Drupal',
        'From' => '"Foo, Bar, and Baz" <from@example.org>',
        'Reply-to' => 'from@example.org',
        'Return-Path' => 'from@example.org',
      ],
    ];

    /** @var \Symfony\Component\Mailer\MailerInterface|\PHPUnit\Framework\MockObject\MockObject */
    $mailer = $this->getMockBuilder(MailerInterface::class)->getMock();
    $mailer->expects($this->once())->method('send')
      ->with(
        $this->logicalAnd(
          $this->callback(fn (Email $email): bool =>
            $email->getHeaders()->get('mime-version')->getBodyAsString() === '1.0'
          ),
          $this->callback(fn (Email $email): bool =>
            $email->getHeaders()->has('content-type') === FALSE
          ),
          $this->callback(fn (Email $email): bool =>
            $email->getHeaders()->has('content-transfer-encoding') === FALSE
          ),
          $this->callback(fn (Email $email): bool =>
            $email->getHeaders()->get('x-mailer')->getBodyAsString() === 'Drupal'
          ),
          $this->callback(fn (Email $email): bool =>
            $email->getHeaders()->get('from')->getBodyAsString() === $expected['from']
          ),
          $this->callback(fn (Email $email): bool =>
            $email->getHeaders()->get('reply-to')->getBodyAsString() === $expected['reply-to']
          ),
          $this->callback(fn (Email $email): bool =>
            $email->getHeaders()->get('to')->getBodyAsString() === $expected['to']
          ),
          $this->callback(fn (Email $email): bool =>
            $email->getHeaders()->get('subject')->getBodyAsString() === $expected['subject']
          ),
          $this->callback(fn (Email $email): bool =>
            $email->getTextBody() === ''
          )
        )
      );

    /** @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject */
    $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

    $plugin = new SymfonyMailer($logger, $mailer);
    $this->assertTrue($plugin->mail($message));
  }

  /**
   * Data provider for testMail().
   */
  public static function providerMail(): array {
    // Verify we use line endings consistent with the PHP mail() function, which
    // changed with PHP 8. See:
    // - https://www.drupal.org/node/3270647
    // - https://bugs.php.net/bug.php?id=81158
    $line_end = "\r\n";

    return [
      'defaults' => [
        [
          'from' => 'from@example.org',
          'reply-to' => 'from@example.org',
          'to' => 'to@example.org',
          'subject' => 'test subject',
        ],
        [
          'from' => '"Foo, Bar, and Baz" <from@example.org>',
          'reply-to' => 'from@example.org',
          'to' => 'to@example.org',
          'subject' => 'test subject',
        ],
      ],
      'line endings in subject' => [
        [
          'from' => 'from@example.org',
          'reply-to' => 'from@example.org',
          'to' => 'to@example.org',
          'subject' => "test\r\nsubject",
        ],
        [
          'from' => '"Foo, Bar, and Baz" <from@example.org>',
          'reply-to' => 'from@example.org',
          'to' => 'to@example.org',
          'subject' => "=?utf-8?Q?test?=$line_end =?utf-8?Q?subject?=",
        ],
      ],
      'multiple comma-separated recipients' => [
        [
          'from' => 'from@example.org',
          'reply-to' => 'from@example.org',
          'to' => 'foo@example.org,bar@example.org',
          'subject' => 'test subject',
        ],
        [
          'from' => '"Foo, Bar, and Baz" <from@example.org>',
          'reply-to' => 'from@example.org',
          'to' => 'foo@example.org, bar@example.org',
          'subject' => 'test subject',
        ],
      ],
    ];
  }

}
