<?php

namespace Drupal\drupal_admin_app\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\Entity\File;
use Drupal\media\Entity\Media;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * Controller for media upload API.
 */
class MediaUploadController extends ControllerBase {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a MediaUploadController object.
   */
  public function __construct(FileSystemInterface $file_system) {
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('file_system')
    );
  }

  /**
   * Upload media file.
   *
   * @param string $media_type
   *   The media type (image, document, audio, video).
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with upload result.
   */
  public function upload($media_type, Request $request) {
    // Check permissions.
    if (!$this->currentUser()->hasPermission('create media')) {
      throw new AccessDeniedHttpException('You do not have permission to upload media.');
    }

    // Validate media type.
    $allowed_types = ['image', 'document', 'audio', 'video'];
    if (!in_array($media_type, $allowed_types)) {
      return new JsonResponse([
        'error' => 'Invalid media type. Allowed: ' . implode(', ', $allowed_types),
      ], 400);
    }

    try {
      // Check Content-Type and handle accordingly.
      $content_type = $request->headers->get('Content-Type');

      if (strpos($content_type, 'multipart/form-data') !== FALSE) {
        // Handle multipart upload.
        $result = $this->handleMultipartUpload($media_type, $request);
      }
      elseif ($content_type === 'application/octet-stream') {
        // Handle octet-stream upload.
        $result = $this->handleOctetStreamUpload($media_type, $request);
      }
      else {
        return new JsonResponse([
          'error' => 'Unsupported Content-Type. Use multipart/form-data or application/octet-stream',
        ], 415);
      }

      return new JsonResponse($result, 201);

    }
    catch (\Exception $e) {
      $this->getLogger('drupal_admin_app')->error('Upload error: @message', [
        '@message' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'error' => 'Error uploading file: ' . $e->getMessage(),
      ], 500);
    }
  }

  /**
   * Handle multipart form data upload.
   */
  private function handleMultipartUpload($media_type, Request $request) {
    $files = $request->files->all();

    if (empty($files)) {
      throw new BadRequestHttpException('No file uploaded.');
    }

    // Get the first file (or you can iterate through all).
    $uploaded_file = reset($files);

    if (!$uploaded_file || !$uploaded_file->isValid()) {
      throw new BadRequestHttpException('Invalid file upload.');
    }

    $filename = $uploaded_file->getClientOriginalName();
    $file_content = file_get_contents($uploaded_file->getPathname());

    return $this->saveMediaFile($media_type, $file_content, $filename);
  }

  /**
   * Handle octet-stream upload.
   */
  private function handleOctetStreamUpload($media_type, Request $request) {
    $file_content = $request->getContent();

    if (empty($file_content)) {
      throw new BadRequestHttpException('No file content provided.');
    }

    // Extract filename from Content-Disposition header.
    $content_disposition = $request->headers->get('Content-Disposition');
    $filename = $this->extractFilename($content_disposition);

    if (empty($filename)) {
      throw new BadRequestHttpException('Filename not provided in Content-Disposition header.');
    }

    return $this->saveMediaFile($media_type, $file_content, $filename);
  }

  /**
   * Save file and create media entity.
   */
  private function saveMediaFile($media_type, $file_content, $filename) {
    // Get field name based on media type.
    $field_name = $this->getFieldName($media_type);

    // Get upload directory.
    $directory = $this->getUploadDirectory($media_type);

    // Prepare directory.
    $this->fileSystem->prepareDirectory(
      $directory,
      FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
    );

    // Save file.
    $destination = $directory . '/' . $filename;
    $file_uri = $this->fileSystem->saveData(
      $file_content,
      $destination,
      FileSystemInterface::EXISTS_RENAME
    );

    if (!$file_uri) {
      throw new \Exception('Failed to save file to disk.');
    }

    // Create file entity.
    $file = File::create([
      'uri' => $file_uri,
      'filename' => basename($file_uri),
      'status' => 1,
    ]);
    $file->save();

    // Create media entity.
    $media = Media::create([
      'bundle' => $media_type,
      'name' => $filename,
      'uid' => $this->currentUser()->id(),
      $field_name => [
        'target_id' => $file->id(),
      ],
    ]);
    $media->save();

    // Return response data.
    return [
      'message' => 'File uploaded successfully',
      'media_id' => $media->id(),
      'file_id' => $file->id(),
      'file_url' => $file->createFileUrl(),
      'media_type' => $media_type,
      'filename' => basename($file_uri),
    ];
  }

  /**
   * Extract filename from Content-Disposition header.
   */
  private function extractFilename($content_disposition) {
    if (empty($content_disposition)) {
      return NULL;
    }

    if (preg_match('/filename[^;=\n]*=["\']?([^"\';]*)["\']?/', $content_disposition, $matches)) {
      return $matches[1];
    }

    return NULL;
  }

  /**
   * Get field name based on media type.
   */
  private function getFieldName($media_type) {
    $field_mapping = [
      'image' => 'field_media_image',
      'document' => 'field_media_document',
      'audio' => 'field_media_audio_file',
      'video' => 'field_media_video_file',
    ];

    return $field_mapping[$media_type] ?? 'field_media_file';
  }

  /**
   * Get upload directory based on media type.
   */
  private function getUploadDirectory($media_type) {
    $directory_mapping = [
      'image' => 'public://images',
      'document' => 'public://documents',
      'audio' => 'public://audio',
      'video' => 'public://videos',
    ];

    return $directory_mapping[$media_type] ?? 'public://media';
  }

}
