<?php

namespace Drupal\drupal_admin_app\Plugin\rest\resource;

use Psr\Log\LoggerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * Provides a resource to manage Log Limit setting.
 *
 * @RestResource(
 *   id = "drupal_admin_app_log_limit",
 *   label = @Translation("Log Limit Config Resource"),
 *   uri_paths = {
 *     "canonical" = "/api/log-limit",
 *     "create" = "/api/log-limit"
 *   }
 * )
 */
class LogLimitResource extends ResourceBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    ConfigFactoryInterface $config_factory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->configFactory = $config_factory;
  }

  /**
   * Creates an instance of the plugin.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   *
   * @return static
   *   Returns an instance of the plugin.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
    $configuration,
    $plugin_id,
    $plugin_definition,
    // REQUIRED.
    $container->getParameter('serializer.formats'),
    // REQUIRED.
    $container->get('logger.factory')->get('drupal_admin_app'),
    $container->get('config.factory')
    );
  }

  /**
   * GET: return current log limit.
   */
  public function get() {
    $log_limit = $this->configFactory
      ->get('drupal_admin_app.settings')
      ->get('log_limit');

    /*return new ResourceResponse([
    'log_limit' => (int) $log_limit,
    ]);*/
    $output = [
      'log_limit' => (int) $log_limit,
    ];

    $response = new ResourceResponse($output);

    // Disable caching completely.
    $response->addCacheableDependency([
      '#cache' => [
        'max-age' => 0,
      ],
    ]);

    return $response;

  }

  /**
   * POST: update log limit.
   */
  public function post(array $data) {
    if (!isset($data['log_limit'])) {
      throw new BadRequestHttpException('Missing log_limit value.');
    }

    $value = (int) $data['log_limit'];

    $this->configFactory
      ->getEditable('drupal_admin_app.settings')
      ->set('log_limit', $value)
      ->save();

    /*return new ResourceResponse([
    'message' => 'Log limit updated successfully.',
    'log_limit' => $value,
    ], 200);*/

    $output = [
      'message' => 'Log limit updated successfully.',
      'log_limit' => $value,
    ];
    $response = new ResourceResponse($output, 200);

    // Disable caching completely.
    $response->addCacheableDependency([
      '#cache' => [
        'max-age' => 0,
      ],
    ]);

    return $response;

  }

}
