<?php

namespace Drupal\drupal_admin_app\Plugin\rest\resource;

use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Psr\Log\LoggerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Provides a REST resource for listing all enabled modules with details.
 *
 * @RestResource(
 *   id = "modules_list_api",
 *   label = @Translation("Modules List API"),
 *   uri_paths = {
 *     "canonical" = "/api/modules/list"
 *   }
 * )
 */
class ModulesListResource extends ResourceBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
    $configuration,
    $plugin_id,
    $plugin_definition,
    $container->getParameter('serializer.formats'),
    $container->get('logger.factory')->get('modules_list_resource'),
     $container->get('module_handler')
    );
  }

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->moduleHandler = $module_handler;

  }

  /**
   * Responds to GET requests.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The response containing the list of modules with pagination.
   */
  public function get() {
    try {
      // ============ GET PAGE PARAMETER START ============
      // Get page parameter from query string (default to 0)
      $request = \Drupal::request();
      $page = $request->query->get('page') ?? 0;
      $page = (int) $page;

      // Get search name parameter.
      $search_name = $request->query->get('name') ?? '';
      $search_name = trim($search_name);
      // Echo "name--".$search_name;exit;
      // Items per page.
      $items_per_page = 10;

      // Calculate offset.
      $offset = $page * $items_per_page;

      \Drupal::logger('drupal_admin_app')->info('📄 Fetching modules - Page: @page, Offset: @offset, Per Page: @per_page', [
        '@page' => $page,
        '@offset' => $offset,
        '@per_page' => $items_per_page,
      ]);
      // ============ GET PAGE PARAMETER END ============
      // Get all modules (both enabled and disabled)
      // $all_modules = $this->moduleHandler->getList();
      $extensionList = \Drupal::service('extension.list.module');
      $all_modules = $extensionList->getList();

      // Get enabled modules.
      $enabled_modules = \Drupal::moduleHandler()->getModuleList();

      // print_r($enabled_modules);exit;
      \Drupal::logger('drupal_admin_app')->info('🔍 Fetching modules list - Total all: @total, Enabled: @enabled', [
        '@total' => count($all_modules),
        '@enabled' => count($enabled_modules),
      ]);

      $modules_data = [];

      foreach ($all_modules as $module_name => $module) {
        // Check if module is enabled.
        $is_enabled = isset($enabled_modules[$module_name]) ? TRUE : FALSE;

        // Only include enabled modules in the response.
        if (!$is_enabled) {
          continue;
        }

        $module_info = $module->info;

        // ============ GET DEPENDENCIES START ============
        // Get modules that this module depends on
        $dependencies = $module_info['dependencies'] ?? [];

        // Clean dependencies (remove version constraints)
        $required_modules = [];
        foreach ($dependencies as $dependency) {
          // Dependencies text".
          $dep_parts = explode(':', $dependency);
          $required_modules[] = $dep_parts[0];
        }

        // Get modules that depend on this module (reverse dependencies)
        $used_by_modules = [];
        foreach ($all_modules as $check_module_name => $check_module) {
          $check_info = $check_module->info;
          if (isset($check_info['dependencies'])) {
            foreach ($check_info['dependencies'] as $dependency) {
              $dep_parts = explode(':', $dependency);
              if ($dep_parts[0] === $module_name && isset($enabled_modules[$check_module_name])) {
                $used_by_modules[] = [
                  'name' => $check_module_name,
                  'label' => $check_info['name'] ?? $check_module_name,
                ];
              }
            }
          }
        }
        // ============ GET DEPENDENCIES END ============
        // Get module details.
        $package = $module_info['package'] ?? 'Other';
        $version = $module_info['version'] ?? 'Unknown';
        $description = $module_info['description'] ?? '';
        $core_version = $module_info['core_version_requirement'] ?? 'Unknown';

        // Build module data.
        $module_data = [
          'name' => $module_name,
          'label' => $module_info['name'] ?? $module_name,
          'description' => $description,
          'version' => $version,
          'package' => $package,
          'core_version_requirement' => $core_version,
          'status' => 'enabled',
          'dependencies' => [
            'count' => count($required_modules),
            'modules' => $required_modules,
          ],
          'used_by' => [
            'count' => count($used_by_modules),
            'modules' => $used_by_modules,
          ],
          'type' => $module_info['type'] ?? 'module',
        ];

        $modules_data[] = $module_data;
      }

      // Sort modules by label for consistent ordering.
      usort($modules_data, function ($a, $b) {
        return strcmp($a['label'], $b['label']);
      });

      // ============ SEARCH/FILTER START ============
      // Filter modules by name if search parameter is provided
      if (!empty($search_name)) {
        // Echo "namey--".$search_name;exit;.
        $search_name_lower = strtolower($search_name);

        $modules_data = array_filter($modules_data, function ($module) use ($search_name_lower) {
          // Search in module name.
          if (strpos(strtolower($module['name']), $search_name_lower) !== FALSE) {
            return TRUE;
          }
          // Search in module label.
          if (strpos(strtolower($module['label']), $search_name_lower) !== FALSE) {
            return TRUE;
          }
          // Search in module description.
          if (strpos(strtolower($module['description']), $search_name_lower) !== FALSE) {
            return TRUE;
          }
          return FALSE;
        });

        // Re-index array after filtering.
        $modules_data = array_values($modules_data);
        // print_r($modules_data);exit;
        \Drupal::logger('drupal_admin_app')->info('🔍 Search filter applied - Search: "@search", Found: @count modules', [
          '@search' => $search_name,
          '@count' => count($modules_data),
        ]);
      }
      // ============ SEARCH/FILTER END ============
      // ============ PAGINATION START ============
      $total_items = count($modules_data);
      $total_pages = ceil($total_items / $items_per_page);

      // Validate page number.
      if ($page < 0) {
        $page = 0;
      }
      if ($page >= $total_pages && $total_pages > 0) {
        $page = $total_pages - 1;
      }

      // Get paginated results.
      $paginated_data = array_slice($modules_data, $offset, $items_per_page);

      \Drupal::logger('drupal_admin_app')->info('📊 Pagination - Page: @page/@total_pages, Items in page: @items/@total', [
        '@page' => $page,
        '@total_pages' => $total_pages,
        '@items' => count($paginated_data),
        '@total' => $total_items,
      ]);
      // ============ PAGINATION END ============
      // print_r($paginated_data);exit;
      // ============ BUILD RESPONSE START ============
      // Build response with metadata
      $response_data = [
        'data' => $paginated_data,
        'pagination' => [
          'page' => $page,
          'items_per_page' => $items_per_page,
          'total_items' => $total_items,
          'total_enabled_modules' => $total_items,
          'total_pages' => $total_pages,
          'items_in_current_page' => count($paginated_data),
          'has_next_page' => ($page + 1) < $total_pages,
          'has_previous_page' => $page > 0,
        ],
        'meta' => [
          'timestamp' => time(),
          'drupal_version' => \Drupal::VERSION,
        ],
      ];

      \Drupal::logger('drupal_admin_app')->info('✅ Modules list generated - Page @current/@total, @total_modules enabled modules', [
        '@current' => $page + 1,
        '@total' => $total_pages,
        '@total_modules' => $total_items,
      ]);

      // Return new ResourceResponse($response_data, 200);
      // Disable caching completely.
      $response = new ResourceResponse($response_data, 200);
      $response->addCacheableDependency([
        '#cache' => [
          'max-age' => 0,
        ],
      ]);
      return $response;
      // ============ BUILD RESPONSE END ============
    }
    catch (\Exception $e) {
      \Drupal::logger('drupal_admin_app')->error('❌ Error fetching modules: @error', [
        '@error' => $e->getMessage(),
      ]);

      return new ResourceResponse([
        'error' => 'Error fetching modules list',
        'message' => $e->getMessage(),
      ], 500);
    }
  }

}
