<?php

namespace Drupal\drupal_admin_app\Plugin\rest\resource;

use Drupal\Core\Database\Connection;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Provides a REST resource for listing all logs with details.
 *
 * @RestResource(
 *   id = "recent_logs_resource",
 *   label = @Translation("Recent Logs REST"),
 *   uri_paths = {
 *     "canonical" = "/api/logs/{action}"
 *   }
 * )
 */
class RecentLogsResource extends ResourceBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    $logger_factory,
    Connection $database,
    ConfigFactoryInterface $config_factory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, [], $logger_factory->get('drupal_admin_app'));
    $this->database = $database;
    $this->configFactory = $config_factory;
  }

  /**
   * Creates an instance of the plugin.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   *
   * @return static
   *   Returns an instance of the plugin.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
    $configuration,
    $plugin_id,
    $plugin_definition,
    $container->get('logger.factory'),
    $container->get('database'),
    $container->get('config.factory')
    );
  }

  /**
   * Responds to GET requests for log-related actions.
   *
   * Returns different log data based on the requested action.
   *
   * @param string $action
   *   The action to perform. Supported values:
   *   - recent: Returns recent log entries.
   *   - types: Returns available log types.
   *   - severity: Returns log severity levels.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The REST response containing the requested log data.
   */
  public function get($action = 'recent') {

    switch ($action) {

      /* ------------------------------
       * 1) /api/logs/recent
       * ------------------------------ */
      case 'recent':
        return $this->getRecentLogs();

      /* ------------------------------
       * 2) /api/logs/types
       * ------------------------------ */
      case 'types':
        return $this->getTypes();

      /* ------------------------------
       * 3) /api/logs/severity
       * ------------------------------ */
      case 'severity':
        return $this->getSeverity();

      default:
        return new ResourceResponse(['error' => 'Invalid action'], 400);
    }
  }

  /**
 * Return recent logs with optional filters .*/
private function getRecentLogs() {
  $request = \Drupal::request();
  $severity = $request->query->get('severity');
  $type = $request->query->get('type');

  // Get the configured log limit.
  $config = $this->configFactory->get('drupal_admin_app.settings');
  $log_limit = $config->get('log_limit') ?: 20;

  $query = $this->database->select('watchdog', 'w')
    ->fields('w')
    ->orderBy('timestamp', 'DESC');

  if ($severity !== NULL) {
    $query->condition('severity', (int) $severity);
  }
  if ($type) {
    $query->condition('type', $type);
  }
  $query->range(0, $log_limit);
  \Drupal::logger('drupal_admin_app')->notice($query->__toString());
  $logs = $query->execute()->fetchAll();

  $output = [];
  foreach ($logs as $log) {
    // Convert stdClass → array.
    $log = (array) $log;

    // Unserialize variables and replace placeholders in message.
    $variables = [];
    if (!empty($log['variables'])) {
      try {
        $unserialized = unserialize($log['variables'], ['allowed_classes' => true]);
        if (is_array($unserialized)) {
          // Convert everything to strings safely.
          $variables = array_map(function($value) {
            if (is_object($value) && method_exists($value, '__toString')) {
              return (string) $value;
            }
            return is_scalar($value) ? (string) $value : '';
          }, $unserialized);
        }
      } catch (\Exception $e) {
        // Log error and continue with empty variables.
        \Drupal::logger('drupal_admin_app')->error('Failed to unserialize log variables: @error', [
          '@error' => $e->getMessage()
        ]);
      }
    }

    // Replace placeholders in the message.
    $log['formatted_message'] = !empty($variables) ? strtr($log['message'], $variables) : $log['message'];
    $log['variables'] = $variables;

    $output[] = $log;
  }

  $response = new ResourceResponse($output);

  // Disable caching completely.
  $response->addCacheableDependency([
    '#cache' => [
      'max-age' => 0,
    ],
  ]);

  return $response;
}

  /**
   * Return list of all distinct log types .*/
  private function getTypes() {
    $types = $this->database->select('watchdog', 'w')
      ->distinct()
      ->fields('w', ['type'])
      ->execute()
      ->fetchCol();

    //return new ResourceResponse(array_values($types));
    $response = new ResourceResponse(array_values($types));

    // Disable caching completely.
    $response->addCacheableDependency([
      '#cache' => [
        'max-age' => 0,
      ],
    ]);

    return $response;
  }

  /**
   * Return list of all distinct severity levels .*/
  private function getSeverity() {
    $severity_numbers = $this->database->select('watchdog', 'w')
      ->distinct()
      ->fields('w', ['severity'])
      ->orderBy('severity')
      ->execute()
      ->fetchCol();

    $severity_output = [];

    foreach ($severity_numbers as $num) {
      $severity_output[] = [
        'value' => (int) $num,
        'label' => RfcLogLevel::getLevels()[$num],
      ];
    }

    //return new ResourceResponse($severity_output);
    $response = new ResourceResponse($severity_output);

    // Disable caching completely.
    $response->addCacheableDependency([
      '#cache' => [
        'max-age' => 0,
      ],
    ]);

    return $response;
  }

}
