<?php

namespace Drupal\drupal_admin_app\Plugin\rest\resource;

use Drupal\Component\Utility\Unserialize;
use Drupal\Core\Database\Connection;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/*

GET /api/top/access-denied
GET /api/top/access-denied?limit=50

GET /api/top/page-not-found
GET /api/top/page-not-found?limit=30

GET /api/top/search-phrases
GET /api/top/search-phrases?limit=25

 */

/**
 * Provides REST Resource for Top Errors and Search Phrases.
 *
 * @RestResource(
 *   id = "top_errors_resource",
 *   label = @Translation("Top Errors and Search REST"),
 *   uri_paths = {
 *     "canonical" = "/api/top/{type}"
 *   }
 * )
 */
class TopErrorsResource extends ResourceBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a TopErrorsResource object.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    $logger_factory,
    Connection $database,
    ConfigFactoryInterface $config_factory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, [], $logger_factory->get('drupal_admin_app'));
    $this->database = $database;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory'),
      $container->get('database'),
      $container->get('config.factory')
    );
  }

  /**
   * Responds to GET requests.
   *
   * @param string $type
   *   The type of data to retrieve (access-denied, page-not-found, search-phrases).
   *
   * @return \Drupal\rest\ResourceResponse
   *   The response containing the requested data.
   */
  public function get($type = 'access-denied') {
    $request = \Drupal::request();
    $limit = $request->query->get('limit');

    // Get default limit from config if not provided.
    if (!$limit) {
      $config = $this->configFactory->get('drupal_admin_app.settings');
      $limit = $config->get('log_limit') ?: 20;
    }

    switch ($type) {
      case 'access-denied':
        return $this->getTopAccessDenied($limit);

      case 'page-not-found':
        return $this->getTopPageNotFound($limit);

      case 'search-phrases':
        return $this->getTopSearchPhrases($limit);

      default:
        return new ResourceResponse(['error' => 'Invalid type. Use: access-denied, page-not-found, or search-phrases'], 400);
    }
  }

  /**
   * Parse variables from watchdog entry.
   * Handles both serialized and JSON formats.
   *
   * @param mixed $variables_data
   *   The variables data from watchdog.
   *
   * @return array
   *   Parsed variables array.
   */
  private function parseVariables($variables_data) {
    if (empty($variables_data)) {
      return [];
    }

    // Try JSON decode first (Drupal 9.3+)
    $decoded = json_decode($variables_data, TRUE);
    if (is_array($decoded)) {
      return $decoded;
    }

    // Fallback to unserialize for older Drupal versions
    if (is_string($variables_data)) {
      try {
        return unserialize($variables_data, ['allowed_classes' => FALSE]) ?: [];
      } catch (\Exception $e) {
        \Drupal::logger('top_errors_resource')->warning('Failed to parse variables: @error', ['@error' => $e->getMessage()]);
        return [];
      }
    }

    return [];
  }

  /**
   * Format message by replacing placeholders with actual values.
   *
   * @param string $message
   *   The message template.
   * @param array $variables
   *   The variables to replace.
   *
   * @return string
   *   The formatted message.
   */
  private function formatMessage($message, array $variables) {
    if (empty($variables)) {
      return $message;
    }

    // Replace all placeholders with their actual values
    foreach ($variables as $key => $value) {
      // Handle different placeholder formats: @key, %key, !key
      $message = str_replace('@' . substr($key, 1), $value, $message);
      $message = str_replace('%' . substr($key, 1), $value, $message);
      $message = str_replace('!' . substr($key, 1), $value, $message);
      // Also try direct replacement
      $message = str_replace($key, $value, $message);
    }

    return $message;
  }

  /**
   * Get top access denied errors.
   *
   * @param int $limit
   *   Maximum number of results to return.
   *
   * @return \Drupal\rest\ResourceResponse
   *   Response with top access denied errors.
   */
  private function getTopAccessDenied($limit) {
    $query = $this->database->select('watchdog', 'w')
      ->fields('w', ['message', 'variables', 'location', 'timestamp'])
      ->condition('type', 'access denied')
      ->orderBy('timestamp', 'DESC')
      ->range(0, $limit);

    $results = $query->execute()->fetchAll();

    $output = [];
    $url_counts = [];

    foreach ($results as $row) {
      // ============ FIX: Parse variables correctly ============
      $variables = $this->parseVariables($row->variables);

      // Extract URL from variables or location
      $url = $variables['@uri']
        ?? $variables['%uri']
        ?? $variables['@path']
        ?? $variables['%path']
        ?? (!empty($row->location) ? parse_url($row->location, PHP_URL_PATH) : 'Unknown');

      // ============ FIX: Format message with actual values ============
      $message = $this->formatMessage($row->message, $variables);

      if (!isset($url_counts[$url])) {
        $url_counts[$url] = [
          'url' => $url,
          'message' => $message,
          'count' => 0,
          'last_occurrence' => 0,
          'last_occurrence_date' => '',
        ];
      }

      $url_counts[$url]['count']++;

      // Track the most recent timestamp
      $timestamp = $row->timestamp ?? time();

      if ($timestamp > $url_counts[$url]['last_occurrence']) {
        $url_counts[$url]['last_occurrence'] = $timestamp;
        $url_counts[$url]['last_occurrence_date'] = date('Y-m-d H:i:s', $timestamp);
      }
    }

    // Sort by count descending
    uasort($url_counts, function ($a, $b) {
      return $b['count'] - $a['count'];
    });

    $output = array_values($url_counts);

    return new ResourceResponse([
      'type' => 'access_denied',
      'total_unique_urls' => count($output),
      'data' => $output,
    ]);
  }

  /**
   * Get top page not found errors.
   *
   * @param int $limit
   *   Maximum number of results to return.
   *
   * @return \Drupal\rest\ResourceResponse
   *   Response with top page not found errors.
   */
  private function getTopPageNotFound($limit) {
    $query = $this->database->select('watchdog', 'w')
      ->fields('w', ['message', 'variables', 'location', 'timestamp'])
      ->condition('type', 'page not found')
      ->orderBy('timestamp', 'DESC')
      ->range(0, $limit);

    $results = $query->execute()->fetchAll();

    $url_counts = [];

    foreach ($results as $row) {
      // ============ FIX: Parse variables correctly ============
      $variables = $this->parseVariables($row->variables);

      // Extract URL from variables or location
      $url = $variables['@uri']
        ?? $variables['%uri']
        ?? $variables['@path']
        ?? $variables['%path']
        ?? (!empty($row->location) ? parse_url($row->location, PHP_URL_PATH) : 'Unknown');

      // ============ FIX: Format message with actual values ============
      $message = $this->formatMessage($row->message, $variables);

      if (!isset($url_counts[$url])) {
        $url_counts[$url] = [
          'url' => $url,
          'message' => $message,
          'count' => 0,
          'last_occurrence' => 0,
          'last_occurrence_date' => '',
        ];
      }

      $url_counts[$url]['count']++;

      // Track the most recent timestamp
      $timestamp = $row->timestamp ?? time();

      if ($timestamp > $url_counts[$url]['last_occurrence']) {
        $url_counts[$url]['last_occurrence'] = $timestamp;
        $url_counts[$url]['last_occurrence_date'] = date('Y-m-d H:i:s', $timestamp);
      }
    }

    // Sort by count descending
    uasort($url_counts, function ($a, $b) {
      return $b['count'] - $a['count'];
    });

    $output = array_values($url_counts);

    $response = new ResourceResponse([
      'type' => 'page_not_found',
      'total_unique_urls' => count($output),
      'data' => $output,
    ]);

    // Disable caching
    $response->addCacheableDependency(['#cache' => ['max-age' => 0]]);
    return $response;
  }

  /**
   * Get top search phrases.
   *
   * @param int $limit
   *   Maximum number of results to return.
   *
   * @return \Drupal\rest\ResourceResponse
   *   Response with top search phrases.
   */
  private function getTopSearchPhrases($limit) {
    $query = $this->database->select('watchdog', 'w')
      ->fields('w', ['message', 'variables', 'timestamp'])
      ->condition('type', 'search')
      ->orderBy('timestamp', 'DESC')
      ->range(0, $limit);

    $results = $query->execute()->fetchAll();

    $search_counts = [];

    foreach ($results as $row) {
      // ============ FIX: Parse variables correctly ============
      $variables = $this->parseVariables($row->variables);

      // Extract search keyword from variables
      $keyword = $variables['@keys']
        ?? $variables['%keys']
        ?? $variables['keys']
        ?? 'Unknown';

      if (!isset($search_counts[$keyword])) {
        $search_counts[$keyword] = [
          'keyword' => $keyword,
          'count' => 0,
          'last_search' => 0,
          'last_search_date' => '',
        ];
      }

      $search_counts[$keyword]['count']++;

      // Track the most recent timestamp
      $timestamp = $row->timestamp ?? time();

      if ($timestamp > $search_counts[$keyword]['last_search']) {
        $search_counts[$keyword]['last_search'] = $timestamp;
        $search_counts[$keyword]['last_search_date'] = date('Y-m-d H:i:s', $timestamp);
      }
    }

    // Sort by count descending
    uasort($search_counts, function ($a, $b) {
      return $b['count'] - $a['count'];
    });

    $output = array_values($search_counts);

    $response = new ResourceResponse([
      'type' => 'search_phrases',
      'total_unique_phrases' => count($output),
      'data' => $output,
    ]);

    // Disable caching
    $response->addCacheableDependency(['#cache' => ['max-age' => 0]]);
    return $response;
  }

}