<?php

namespace Drupal\drupal_content_repository\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for rendering Drupal entities via repository endpoints.
 */
class DrupalContentRepositoryController extends ControllerBase {

  /**
   * Renders an entity using the specified parameters.
   *
   * @param string $entity_type
   *   The machine name of the entity type.
   * @param string $id
   *   The ID of the entity to render.
   * @param string $view_mode
   *   The view mode to use for rendering.
   * @param string $language
   *   The language code to render the entity in.
   * @param string $check_access
   *   Whether to check access permissions.
   *
   * @return array
   *   A render array.
   */
  public function renderEntity($entity_type, $id, $view_mode = '', $language = '', $check_access = 'true') {
    // Handle special case for user entities - always return unavailable
    if ($entity_type === 'user') {
      return [
        '#theme' => 'drupal_content_repository_entity',
        '#entity_type' => $entity_type,
        '#id' => $id,
        '#view_mode' => $view_mode,
        '#language' => $language,
        '#check_access' => $check_access,
        '#error_message' => 'User info unavailable.',
        '#error_class' => 'user-unavailable',
      ];
    }

    // Check node publishing status if setting is enabled
    if ($entity_type === 'node') {
      $config = \Drupal::config('drupal_content_repository.settings');
      $check_published = $config->get('check_node_published') ?? FALSE;
      
      if ($check_published) {

        $node = $node = \Drupal\node\Entity\Node::load($id);
        if ($node !== FALSE) {
          $status = $node->isPublished();
        }
        
        // If no record found or status is not 1 (published)
        if ($status === FALSE || $status != 1) {
          return [
            '#theme' => 'drupal_content_repository_entity',
            '#entity_type' => $entity_type,
            '#id' => $id,
            '#view_mode' => $view_mode,
            '#language' => $language,
            '#check_access' => $check_access,
            '#error_message' => 'Node unavailable.',
            '#error_class' => 'node-unavailable',
          ];

        }
      }
    }

    // Convert string 'true'/'false' to boolean
    $check_access_bool = $check_access === 'true' || $check_access === '1';

    return [
      '#theme' => 'drupal_content_repository_entity',
      '#entity_type' => $entity_type,
      '#id' => $id,
      '#view_mode' => $view_mode ?: NULL,
      '#language' => $language ?: NULL,
      '#check_access' => $check_access_bool,
    ];
  }

}