<?php

namespace Drupal\drupal_metrics\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Url;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Link;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\StringTranslation\ByteSizeMarkup;

/**
 * Controller for the Drupal Metrics module.
 *
 * Provides pages under Reports > Metrics, including:
 * - Database metrics (table sizes, total size).
 * - Entity and bundle overviews with field statistics.
 * - Links to detailed bundle field information.
 */
class MetricsController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructs a new MetricsController.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager service.
   */
  public function __construct(Connection $database, EntityFieldManagerInterface $entityFieldManager) {
    $this->database = $database;
    $this->entityFieldManager = $entityFieldManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * Builds the database metrics report page.
   *
   * @return array
   *   A render array containing the database metrics summary and table.
   */
  public function metricsPage() {
    $build['#attached']['library'][] = 'drupal_metrics/drupal_metrics.styles';

    $header = [
      'name' => ['data' => $this->t('Table Name'), 'field' => 'table_name'],
      'size' => ['data' => $this->t('Size (MB)'), 'field' => 'size', 'sort' => 'desc'],
    ];

    $query = $this->database->select('information_schema.tables', 't');
    $query->addField('t', 'table_name', 'name');
    $query->addExpression('ROUND((data_length + index_length), 2)', 'size');
    $query->condition('table_schema', $this->database->getConnectionOptions()['database']);
    $query = $query->extend('Drupal\Core\Database\Query\TableSortExtender');
    $query->orderByHeader($header);

    $results = $query->execute()->fetchAll();

    $rows = [];
    $totalSize = 0;

    foreach ($results as $row) {
      $rows[] = [
        'data' => [
          $row->name,
          [
            'data' => ByteSizeMarkup::create($row->size),
            'sort' => $row->size,
          ],
        ],
      ];
      $totalSize += $row->size;
    }

    // Summary block.
    $summary = [
      '#type' => 'container',
      '#attributes' => ['class' => ['drupal-metrics-summary']],
      'db_size' => [
        '#markup' => '<strong>' . $this->t('Total Database Size:') . '</strong> ' . ByteSizeMarkup::create($totalSize),
      ],
    ];

    $build['summary'] = $summary;
    $build['table'] = [
      '#theme' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No database tables found.'),
    ];

    return $build;
  }

  /**
   * Entities overview tab.
   */
  public function entityOverview() {
    $header = [
      ['data' => $this->t('Entity Type')],
      ['data' => $this->t('Bundle Count')],
    ];

    $rows = [];
    foreach ($this->entityTypeManager()->getDefinitions() as $entity_type_id => $entity_type) {
      if ($entity_type->entityClassImplements(ContentEntityInterface::class)) {
        $bundle_entity_type = $entity_type->getBundleEntityType();
        $bundle_count = 0;

        if ($bundle_entity_type) {
          $bundle_storage = $this->entityTypeManager()->getStorage($bundle_entity_type);
          $bundle_count = $bundle_storage->getQuery()
            ->accessCheck(FALSE)
            ->count()
            ->execute();
        }

        $rows[] = [
          [
            'data' => Link::fromTextAndUrl(
              $entity_type->getLabel(),
              Url::fromRoute('drupal_metrics.entity_bundles', ['entity_type_id' => $entity_type_id])
            )->toRenderable(),
          ],
          ['data' => $bundle_count],
        ];
      }
    }

    return [
      '#theme' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No content entities found.'),
    ];
  }

  /**
   * Bundles of a given entity type.
   */
  public function entityBundles($entity_type_id) {
    $entity_type = $this->entityTypeManager()->getDefinition($entity_type_id);
    $bundle_entity_type = $entity_type->getBundleEntityType();

    $header = [
      $this->t('Bundle'),
      $this->t('Base Fields'),
      $this->t('Additional Fields'),
    ];

    $rows = [];
    if ($bundle_entity_type) {
      $bundles = $this->entityTypeManager()->getStorage($bundle_entity_type)->loadMultiple();

      foreach ($bundles as $bundle_id => $bundle) {
        $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);

        $base_fields_count = 0;
        $additional_fields_count = 0;

        foreach ($field_definitions as $field_definition) {
          if ($field_definition->getFieldStorageDefinition()->isBaseField()) {
            $base_fields_count++;
          }
          else {
            $additional_fields_count++;
          }
        }

        $rows[] = [
          [
            'data' => Link::fromTextAndUrl(
              $bundle->label(),
              Url::fromRoute('drupal_metrics.entity_bundle_fields', [
                'entity_type_id' => $entity_type_id,
                'bundle' => $bundle_id,
              ])
            )->toRenderable(),
          ],
          ['data' => $base_fields_count],
          ['data' => $additional_fields_count],
        ];
      }
    }

    return [
      '#theme' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No bundles found for this entity type.'),
    ];
  }

  /**
   * Fields of a given bundle.
   */
  public function bundleFields($entity_type_id, $bundle) {
    $entity_type = $this->entityTypeManager()->getDefinition($entity_type_id);

    $base_fields = $this->entityFieldManager->getBaseFieldDefinitions($entity_type_id);
    $field_configs = $this->entityTypeManager()
      ->getStorage('field_config')
      ->loadByProperties([
        'entity_type' => $entity_type_id,
        'bundle' => $bundle,
      ]);

    $base_field_count = count($base_fields);
    $additional_field_count = count($field_configs);

    $storage = $this->entityTypeManager()->getStorage($entity_type_id);

    $entity_count = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition($entity_type->getKey('bundle'), $bundle)
      ->count()
      ->execute();

    $translation_count = 0;
    if ($entity_type->isTranslatable()) {
      $entity_ids = $storage->getQuery()
        ->accessCheck(FALSE)
        ->condition($entity_type->getKey('bundle'), $bundle)
        ->execute();

      if (!empty($entity_ids)) {
        $entities = $storage->loadMultiple($entity_ids);
        foreach ($entities as $entity) {
          foreach ($entity->getTranslationLanguages(TRUE) as $langcode => $language) {
            if ($langcode !== $entity->language()->getId()) {
              $translation_count++;
            }
          }
        }
      }
    }

    $summary_header = [
      $this->t('Bundle'),
      $this->t('Entity Count'),
      $this->t('Translations'),
      $this->t('Base Fields'),
      $this->t('Additional Fields'),
    ];

    $summary_rows = [
      [$bundle, $entity_count, $translation_count, $base_field_count, $additional_field_count],
    ];

    $field_header = [
      ['data' => $this->t('Field Name')],
      ['data' => $this->t('Label')],
      ['data' => $this->t('Type')],
      ['data' => $this->t('Cardinality')],
    ];

    $default_rows = [];
    foreach ($base_fields as $field_name => $definition) {
      $default_rows[] = [
        $field_name,
        $definition->getLabel(),
        $definition->getType(),
        $definition->getCardinality() == -1 ? $this->t('Unlimited') : $definition->getCardinality(),
      ];
    }

    $additional_rows = [];
    foreach ($field_configs as $field_config) {
      $additional_rows[] = [
        $field_config->getName(),
        $field_config->label(),
        $field_config->getType(),
        $field_config->getFieldStorageDefinition()->getCardinality() == -1
          ? $this->t('Unlimited')
          : $field_config->getFieldStorageDefinition()->getCardinality(),
      ];
    }

    return [
      'summary' => [
        '#theme' => 'table',
        '#header' => $summary_header,
        '#rows' => $summary_rows,
        '#caption' => $this->t('Bundle summary'),
      ],
      'default' => [
        '#theme' => 'table',
        '#header' => $field_header,
        '#rows' => $default_rows,
        '#caption' => $this->t('Default (base) fields'),
        '#empty' => $this->t('No base fields found.'),
      ],
      'additional' => [
        '#theme' => 'table',
        '#header' => $field_header,
        '#rows' => $additional_rows,
        '#caption' => $this->t('Additional (configurable) fields'),
        '#empty' => $this->t('No additional fields found.'),
      ],
    ];
  }

  /**
   * Page title for entity bundles.
   */
  public function entityBundlesTitle($entity_type_id) {
    $entity_type = $this->entityTypeManager()->getDefinition($entity_type_id);
    return $this->t('Bundles for @entity', ['@entity' => $entity_type->getLabel()]);
  }

  /**
   * Page title for bundle fields.
   */
  public function bundleFieldsTitle($entity_type_id, $bundle) {
    $entity_type = $this->entityTypeManager()->getDefinition($entity_type_id);
    return $this->t('Fields for @bundle bundle of @entity', [
      '@bundle' => $bundle,
      '@entity' => $entity_type->getLabel(),
    ]);
  }

}
