<?php

namespace Drupal\purview_unified_catalog_ui\Controller;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient;
use Drupal\drupal_purview\Utility\PurviewHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for displaying and handling governance domains.
 *
 * Provides:
 *  - A page render callback that builds a hierarchy of domains.
 *  - An AJAX callback that returns metadata for a selected domain.
 */
class GovernanceDomainsListingController extends ControllerBase {

  /**
   * The Purview governance domain API client.
   *
   * @var \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
   */
  protected PurviewGovernanceDomainApiClient $governanceDomainApiClient;

  /**
   * The module extension list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $extensionListModule;

  /**
   * Constructs the controller.
   */
  public function __construct(
    PurviewGovernanceDomainApiClient $governanceDomainApiClient,
    ModuleExtensionList $extensionListModule,
  ) {
    $this->governanceDomainApiClient = $governanceDomainApiClient;
    $this->extensionListModule = $extensionListModule;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('drupal_purview.governance_domain_api'),
      $container->get('extension.list.module')
    );
  }

  /**
   * Returns the page render array.
   */
  public function page(): array {
    // Get domains.
    $domains = $this->governanceDomainApiClient->getGovernanceDomains();

    // Build the hierarchy.
    if (is_array($domains)) {
      $tree = $this->governanceDomainApiClient->buildGovernanceDomainHierarchy($domains);
    }

    // Get default right-side content.
    $config = $this->config('purview_unified_catalog_ui.settings');
    $default_panel = $config->get('governance_panel_default');

    $default_render = [
      '#type' => 'processed_text',
      '#text' => $default_panel['value'] ?? '',
      '#format' => $default_panel['format'] ?? 'basic_html',
    ];

    return [
      '#theme' => 'governance_domains_wrapper',
      '#tree' => $tree ?? '',
      '#default_content' => $default_render,
      '#attached' => [
        'library' => [
          'core/drupal.ajax',
          'purview_unified_catalog_ui/governance_domains_listing',
        ],
      ],
    ];
  }

  /**
   * AJAX callback: returns domain detail markup.
   */
  public function detail(string $domain_id): AjaxResponse {
    // Get domains.
    $domains = $this->governanceDomainApiClient->getGovernanceDomains();

    $domain = NULL;
    foreach ($domains as $item) {
      if ($item['id'] === $domain_id) {
        $domain = $item;
        break;
      }
    }

    if (!$domain) {
      $content = [
        '#type' => 'container',
        '#attributes' => [
          'id' => 'domain-detail-target',
          'class' => ['governance-domains-right'],
        ],
        '#markup' => $this->t('Domain not found.'),
      ];
    }
    else {
      $id = $domain['id'];
      $name = $domain['name'];
      $desc = strip_tags($domain['description'] ?? '');
      $status = $domain['status'];
      $status_class = strtolower($status);

      // Domain first letter.
      $first_letter = strtoupper($name[0]);
      $colors = PurviewHelper::generateLetterColors($first_letter);
      $border_color = $colors['border'];
      $background_color = $colors['background'];

      // Get domain parent.
      $parent_name = $this->governanceDomainApiClient->getDomainParent($id);
      if (!is_null($parent_name)) {
        $parent_first_letter = strtoupper($parent_name[0]);
        $parent_colors = PurviewHelper::generateLetterColors($parent_first_letter);
        $parent_border_color = $parent_colors['border'];
        $parent_background_color = $parent_colors['background'];
      }

      // Build type output.
      $type = $domain['type'];
      if (!empty($type)) {
        // Insert space before every uppercase letter, except first.
        $type_data = preg_replace('/(?<!^)([A-Z])/', ' $1', $type);
        // Lowercase the tail, keep first word capitalized.
        $type_data = ucfirst(strtolower($type_data));
      }

      // Health actions.
      $health_actions = $this->governanceDomainApiClient->getGovernanceDomainHealthActionsCount($id);
      if ($health_actions <= 0) {
        $health_actions = "<div class='value muted'>" . t('No active actions') . "</div>";
      }
      $health_section = "<div class='item'>
                          <div class='label'>" . t('Health actions') . "</div>
                          <div class='value'>{$health_actions}</div>
                        </div>";

      // Get business concepts information.
      $data_product_count = $this->governanceDomainApiClient->getDataProductCountForDomain($id);
      $glossary_term_count = $this->governanceDomainApiClient->getGlossaryTermCountForDomain($id);
      $module_path = $this->extensionListModule->getPath('purview_unified_catalog_ui');

      // Data quality score.
      $score = $this->governanceDomainApiClient->getAverageDataQualityScoreByKey('businessDomainId', $id);
      if (empty($score)) {
        $score_array = [
          'score' => NULL,
          'desc' => $this->t('No score available'),
          'class' => 'muted',
        ];
      }
      else {
        $score_desc = $this->governanceDomainApiClient->getDataQualityLabel($score);
        $score_array = [
          'score' => $score,
          'desc' => $score_desc,
          'class' => 'score--' . strtolower($score_desc),
        ];
      }

      $content = [
        '#theme' => 'governance_domains_metadata',
        '#domain' => [
          'id' => $id,
          'name' => $name,
          'first_letter' => $first_letter,
          'border_color' => $border_color,
          'background_color' => $background_color,
          'description' => $desc,
        ],
        '#parent' => !empty($parent_name) ? [
          'name' => $parent_name,
          'first_letter' => $parent_first_letter,
          'border_color' => $parent_border_color,
          'background_color' => $parent_background_color,
        ] : NULL,
        '#type' => $type_data ?? '',
        '#status' => $status,
        '#status_class' => $status_class,
        '#score' => $score_array,
        '#health_actions' => $health_actions,
        '#data_product_count' => $data_product_count,
        '#glossary_term_count' => $glossary_term_count,
        '#module_path' => $module_path,
      ];
    }

    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand(
      '#domain-detail-target',
      $content
    ));

    return $response;
  }

}
