<?php

namespace Drupal\purview_unified_catalog_ui\Controller;

use Drupal\Component\Datetime\Time;
use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Url;
use Drupal\purview_unified_catalog_ui\Form\UnifiedProductSearchForm;
use Drupal\purview_unified_catalog_ui\Utility\UnifiedCatalogHelper;
use Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient;
use Drupal\drupal_purview\Utility\PurviewHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for displaying the unified product search page and results.
 */
class UnifiedProductSearchController extends ControllerBase {

  /**
   * The governance domain API client.
   *
   * @var \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
   */
  protected PurviewGovernanceDomainApiClient $governanceDomainApiClient;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\Time
   */
  protected Time $time;

  /**
   * The Purview helper service.
   *
   * @var \Drupal\drupal_purview\Utility\PurviewHelper
   */
  protected PurviewHelper $purviewHelper;

  /**
   * The Purview Unified Catalog UI helper service.
   *
   * @var \Drupal\purview_unified_catalog_ui\Utility\UnifiedCatalogHelper
   */
  protected UnifiedCatalogHelper $unifiedCatalogHelper;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Constructs the UnifiedProductSearchController.
   */
  public function __construct(
    FormBuilderInterface $formBuilder,
    PurviewGovernanceDomainApiClient $governanceDomainApiClient,
    ConfigFactoryInterface $configFactory,
    DateFormatterInterface $dateFormatter,
    Time $time,
    PurviewHelper $purviewHelper,
    RequestStack $requestStack,
    UnifiedCatalogHelper $unifiedCatalogHelper,
  ) {
    $this->formBuilder = $formBuilder;
    $this->governanceDomainApiClient = $governanceDomainApiClient;
    $this->configFactory = $configFactory;
    $this->dateFormatter = $dateFormatter;
    $this->time = $time;
    $this->purviewHelper = $purviewHelper;
    $this->requestStack = $requestStack;
    $this->unifiedCatalogHelper = $unifiedCatalogHelper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('form_builder'),
      $container->get('drupal_purview.governance_domain_api'),
      $container->get('config.factory'),
      $container->get('date.formatter'),
      $container->get('datetime.time'),
      $container->get('drupal_purview.helper'),
      $container->get('request_stack'),
      $container->get('purview_unified_catalog_ui.helper'),
    );
  }

  /**
   * Renders the product search page with form and optional results.
   */
  public function page(): array {
    // Get query strings.
    $request = $this->requestStack->getCurrentRequest();
    $keywords = Html::escape((string) $request->query->get('keywords'));
    $domain = Html::escape((string) ($request->query->get('domain') ?? 'all'));
    $attributes = Html::escape((string) ($request->query->get('attributes') ?? 'all'));
    $owner = Html::escape((string) ($request->query->get('owner') ?? 'all'));

    // Get settings.
    $settings = $this->configFactory->get('purview_unified_catalog_ui.settings');
    $display_form = $settings->get('display_search_forms');

    $intro = $settings->get('products_intro');
    $intro_render = [
      '#type' => 'processed_text',
      '#text' => $intro['value'],
      '#format' => $intro['format'],
      '#prefix' => '<div class="glossary-intro">',
      '#suffix' => '</div>',
    ];

    if ($display_form) {
      return [
        '#type' => 'container',
        '#attributes' => ['class' => ['product-search-page']],
        '#prefix' => '<div id="unified-product-offcanvas-wrapper"></div>',
        'form_wrapper' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['product-search-form']],
          'form' => $this->formBuilder->getForm(UnifiedProductSearchForm::class),
        ],
        'intro' => $intro_render,
        'results_wrapper' => $this->buildSearchResults($keywords, $domain, $owner, $attributes),
      ];
    }
    else {
      return [
        '#type' => 'container',
        '#attributes' => ['class' => ['product-search-page']],
        '#prefix' => '<div id="unified-product-offcanvas-wrapper"></div>',
        'intro' => $intro_render,
        'results_wrapper' => $this->buildSearchResults($keywords, $domain, $owner, $attributes),
      ];
    }
  }

  /**
   * Builds the searchable data-product results view (list mode).
   *
   * Fetches data products from Purview for the given filters (keywords, domain,
   * owner, and managed-attribute value) and renders an A–Z grouped results list
   * with an anchor bar, a summary count, and rows themed by
   * 'unified_catalog_product_result'.
   *
   * @param string|null $keywords
   *   Optional free-text query to match product names/descriptions.
   * @param string $domains
   *   Governance domain ID to filter by, or 'all' to search all domains.
   * @param mixed $owner
   *   Optional owner identifier to filter by; pass 'all' to disable.
   * @param mixed $attributes
   *   Optional managed-attribute filter in the form "Group.Field|Value";
   *   pass 'all' to disable.
   *
   * @return array
   *   A render array for the grouped results list. If no matches are found,
   *   returns container with "No results found." message and attached
   *   libraries.
   */
  protected function buildSearchResults(?string $keywords, string $domains, $owner, $attributes = 'all'): array {
    $settings = $this->configFactory->get('purview_unified_catalog_ui.settings');
    $keywords = $keywords ?? '';
    $results = [];

    if (empty($domains) || $domains === 'all') {
      $all_domains = $this->governanceDomainApiClient->getGovernanceDomains();

      if (is_array($all_domains)) {
        foreach ($all_domains as $domain) {
          $products = $this->governanceDomainApiClient->getDataProductsByDomain($domain['id'], $keywords, $owner, $attributes);
          $results = array_merge($results, $products['value']);
        }
      }
    }
    else {
      $products = $this->governanceDomainApiClient->getDataProductsByDomain($domains, $keywords, $owner, $attributes);
      $results = $products['value'];
    }

    if (empty($results)) {
      return [
        '#type' => 'container',
        '#attributes' => ['class' => ['empty-results']],
        '#markup' => $this->t('No results found.'),
        '#attached' => [
          'library' => [
            'purview_unified_catalog_ui/unified_product_search',
          ],
        ],
      ];
    }

    // Group by first character.
    $grouped = [];
    foreach ($results as $product) {
      $name = $product['name'] ?? '';
      $firstChar = strtoupper(mb_substr($name, 0, 1));

      if (ctype_alpha($firstChar)) {
        $key = $firstChar;
      }
      elseif (ctype_digit($firstChar)) {
        $key = '1';
      }
      elseif ($firstChar !== '') {
        $key = '#';
      }
      else {
        continue;
      }

      $grouped[$key][] = $product;
    }

    // Sort groups and names inside each group.
    ksort($grouped);
    foreach ($grouped as &$products) {
      usort($products, fn($a, $b) => strcasecmp($a['name'], $b['name']));
    }
    unset($products);

    $all_sections = array_merge(['#', '1'], range('A', 'Z'));
    $output = [
      '#type' => 'container',
      '#attributes' => ['class' => ['product-search-results']],
      '#attached' => [
        'library' => [
          'core/drupal.dialog.ajax',
          'purview_unified_catalog_ui/unified_product_search',
          'purview_unified_catalog_ui/clear_search',
        ],
      ],
      'summary' => [
        '#markup' => $this->formatPlural(count($results), '@count result found.', '@count results found.'),
        '#prefix' => '<div class="product-search-summary">',
        '#suffix' => '</div>',
      ],
      'anchors' => [
        '#prefix' => '<section class="product-anchor-links"><div class="nav-wrapper"><h2 class="visually-hidden">' . $this->t('Data product navigation') . '</h2>',
        '#suffix' => '</div></section>',
        '#markup' => implode(' ', array_map(function ($char) use ($grouped) {
          return isset($grouped[$char])
            ? "<a href=\"#letter-{$char}\">{$char}</a>"
            : "<span class=\"product-anchor-disabled\">{$char}</span>";
        }, $all_sections)),
      ],
    ];

    // Get domain list.
    $domain_lookup = [];
    foreach ($this->governanceDomainApiClient->getGovernanceDomains() as $d) {
      $domain_lookup[$d['id']] = $d['name'];
    }

    $attribute_label_raw = (string) ($settings->get('attribute_filter_label') ?? '');
    $attribute_label = $attribute_label_raw !== ''
      ? Html::escape($attribute_label_raw)
      : $this->t('- Attribute -');

    foreach ($grouped as $char => $products) {
      $section_id = "letter-$char";
      $output[$section_id . '_heading'] = [
        '#markup' => "<div class='product-section' id=\"$section_id\">$char</div>",
      ];

      $output[$section_id . '_row_headings'] = [
        '#markup' => '<div class="row-header">
                        <div class="column">' . t('Data product name') . '</div>
                        <div class="column hide-mobile">' . t('Type') . '</div>
                        <div class="column">' . t('Governance domain') . '</div>
                        <div class="column">' . t('Data quality score') . '</div>
                        <div class="column hide-mobile">' . t('Last updated') . '</div>
                        <div class="column">' . $attribute_label . '</div>
                      </div>',
      ];

      foreach ($products as $product) {
        // Get product render data.
        $render_data = $this->prepareProductRenderData($product, $domain_lookup);

        $output["{$section_id}_item_" . $product['id']] = [
          '#theme' => 'unified_catalog_product_result',
          '#link' => $render_data['link'],
          '#product' => $product,
          '#name' => $product['name'],
          '#status' => $product['status'] ?? '',
          '#type' => $render_data['type'],
          '#domain' => $render_data['domain'],
          '#updated' => $render_data['updated'],
          '#asset_count' => $render_data['asset_count'],
          '#owners' => $render_data['owners'],
          '#score' => $render_data['score'],
          '#attributes_text' => $render_data['attributes_text'],
        ];
      }
    }

    return $output;
  }

  /**
   * Prepares common render data for a data product.
   *
   * This method builds a render array for the product link, formats the domain
   * metadata, and formats the last modified timestamp.
   *
   * The returned array is intended to be used in both list and tree views to
   * standardize how data products are rendered using the
   * unified_catalog_product_result theme template.
   *
   * @param array $product
   *   The data product data as returned by the Purview API.
   * @param array $domain_lookup
   *   An associative array of domain IDs to domain names.
   *
   * @return array
   *   An array containing:
   *   - link: Render array for the offcanvas product link.
   *   - type: The type of data product.
   *   - domain: Associative array with domain name and styling info.
   *   - updated: Formatted last modified date string.
   *   - asset_count: Number of assets related to the data product.
   *   - owners: The owners of this data product.
   */
  protected function prepareProductRenderData(array $product, array $domain_lookup): array {
    // Link.
    $link = [
      '#type' => 'link',
      '#title' => $product['name'],
      '#url' => Url::fromRoute('purview_unified_catalog_ui.product_metadata_offcanvas', ['guid' => $product['id']]),
      '#options' => [
        'attributes' => [
          'class' => ['use-ajax', 'data-product-link'],
        ],
      ],
    ];

    // Updated time.
    // Updated date.
    $modified_date = $product['systemData']['lastModifiedAt'] ?? NULL;
    $updated_text = $modified_date
      ? $this->dateFormatter->format((new \DateTime($modified_date))->getTimestamp(), 'custom', 'n/j/Y, g:i A')
      : $this->t('Updated date unavailable');

    // Type.
    $type = str_replace('Or', '/', $product['type']);

    // Domain.
    $domain_name = $domain_lookup[$product['domain']] ?? '';
    $first_letter = strtoupper($domain_name[0] ?? '?');
    $colors = $this->purviewHelper->generateLetterColors($first_letter);
    $domain_info = [
      "name" => $domain_name,
      "first_letter" => $first_letter,
      "background_color" => $colors['background'],
      "border_color" => $colors['border'],
    ];

    // Asset count.
    $assets = $product['additionalProperties']['assetCount'];

    // Owners.
    $owners = $this->purviewHelper->renderOwnersHorizontal($product['contacts']['owner']);

    // Data quality score.
    $score = $this->governanceDomainApiClient->getAverageDataQualityScoreByKey('dataProductId', $product['id']);
    if (empty($score)) {
      $score_array = [
        'score' => NULL,
        'desc' => $this->t('--'),
        'class' => 'muted',
      ];
    }
    else {
      $score_desc = $this->governanceDomainApiClient->getDataQualityLabel($score);
      $score_array = [
        'score' => $score,
        'desc' => $score_desc,
        'class' => 'score--' . strtolower($score_desc),
      ];
    }

    // Attributes.
    $attributes = $this->unifiedCatalogHelper->extractManagedAttributeValues($product);
    $attribute_text = !empty($attributes) ? implode(', ', $attributes) : '--';

    return [
      'link' => $link,
      'type' => $type,
      'domain' => $domain_info,
      'updated' => $updated_text,
      'asset_count' => $assets,
      'owners' => $owners,
      'score' => $score_array,
      'attributes_text' => $attribute_text,
    ];
  }

}
