<?php

namespace Drupal\purview_unified_catalog_ui\Controller;

use Drupal\Component\Utility\DeprecationHelper;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenOffCanvasDialogCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\drupal_purview\Service\GraphApiClient;
use Drupal\drupal_purview\Service\PurviewClassicTypesApiClient;
use Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient;
use Drupal\drupal_purview\Utility\PurviewHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for displaying glossary term metadata in an off-canvas panel.
 */
class UnifiedTermMetadataController extends ControllerBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * The Graph API client.
   *
   * @var \Drupal\drupal_purview\Service\GraphApiClient
   */
  protected GraphApiClient $graphApiClient;

  /**
   * The classic types API client.
   *
   * @var \Drupal\drupal_purview\Service\PurviewClassicTypesApiClient
   */
  protected PurviewClassicTypesApiClient $classicTypesApiClient;

  /**
   * The governance domain API client.
   *
   * @var \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
   */
  protected PurviewGovernanceDomainApiClient $governanceDomainApiClient;

  /**
   * The Purview helper service.
   *
   * @var \Drupal\drupal_purview\Utility\PurviewHelper
   */
  protected PurviewHelper $purviewHelper;

  /**
   * The module extension list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $extensionListModule;

  /**
   * Constructs the UnifiedTermMetadataController.
   */
  public function __construct(
    RendererInterface $renderer,
    GraphApiClient $graphApiClient,
    PurviewClassicTypesApiClient $classicTypesApiClient,
    PurviewGovernanceDomainApiClient $governanceDomainApiClient,
    PurviewHelper $purviewHelper,
    ConfigFactoryInterface $configFactory,
    ModuleExtensionList $extensionListModule,
  ) {
    $this->renderer = $renderer;
    $this->graphApiClient = $graphApiClient;
    $this->classicTypesApiClient = $classicTypesApiClient;
    $this->governanceDomainApiClient = $governanceDomainApiClient;
    $this->purviewHelper = $purviewHelper;
    $this->configFactory = $configFactory;
    $this->extensionListModule = $extensionListModule;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('renderer'),
      $container->get('drupal_purview.graph_api_client'),
      $container->get('drupal_purview.api_client'),
      $container->get('drupal_purview.governance_domain_api'),
      $container->get('drupal_purview.helper'),
      $container->get('config.factory'),
      $container->get('extension.list.module'),
    );
  }

  /**
   * Opens the glossary term metadata in an off-canvas panel.
   *
   * @param string $guid
   *   The GUID of the glossary term.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response with off-canvas content.
   */
  public function openOffcanvas(string $guid): AjaxResponse {
    $title = $this->governanceDomainApiClient->getGlossaryTermMetadata($guid)['name'] ?? $this->t('Glossary term metadata');

    $build = [
      '#type' => 'off_canvas',
      '#title' => $title,
      '#attributes' => ['class' => ['glossary-metadata-dialog']],
      'content' => $this->buildTermMetadataRenderArray($guid),
    ];

    $response = new AjaxResponse();
    $response->addCommand(new OpenOffCanvasDialogCommand(
      $title,
      $build,
      ['dialogClass' => 'glossary-metadata-offcanvas']
    ));
    return $response;
  }

  /**
   * Returns a full-page render array for a glossary term.
   *
   * @param string $guid
   *   The GUID of the glossary term.
   *
   * @return array
   *   A render array for displaying the glossary term metadata.
   */
  public function page(string $guid): array {
    $title = $this->governanceDomainApiClient->getGlossaryTermMetadata($guid)['name'] ?? $this->t('Glossary term metadata');
    return [
      '#title' => $title,
      'content' => $this->buildTermMetadataRenderArray($guid),
    ];
  }

  /**
   * Builds the render array for displaying a glossary term's metadata.
   *
   * @param string $guid
   *   The GUID of the glossary term.
   *
   * @return array
   *   A render array representing the metadata view.
   */
  protected function buildTermMetadataRenderArray(string $guid): array {
    $module_path = $this->extensionListModule->getPath('purview_unified_catalog_ui');
    $term = $this->governanceDomainApiClient->getGlossaryTermMetadata($guid);
    $title = $term['name'] ?? $this->t('Glossary term metadata');
    $description = $term['description'] ?? '';
    $status = $term['status'];

    // Authoring information.
    $timezone = $this->configFactory->get('system.date')->get('timezone.default');
    $create_time = (new \DateTime($term['systemData']['createdAt'], new \DateTimeZone('UTC')))
      ->setTimezone(new \DateTimeZone($timezone))
      ->format('n/j/Y, g:i A');
    $update_time = (new \DateTime($term['systemData']['lastModifiedAt'], new \DateTimeZone('UTC')))
      ->setTimezone(new \DateTimeZone($timezone))
      ->format('n/j/Y, g:i A');

    $created_by = $this->graphApiClient->resolveUserNameByGuid($term['systemData']['createdBy']);
    $updated_by = $this->graphApiClient->resolveUserNameByGuid($term['systemData']['lastModifiedBy']);

    $authoring_info = [
      'create_time' => $create_time,
      'update_time' => $update_time,
      'created_by' => $created_by,
      'updated_by' => $updated_by,
    ];

    // Owners.
    $owner_output = '';
    if (!empty($term['contacts']['owner'])) {
      $owner_output = $this->purviewHelper->renderOwnersSmall($term['contacts']['owner']);
    }

    // Domain info.
    $domain_id = $term['domain'];
    $domain = $this->governanceDomainApiClient->getGovernanceDomainMetadata($domain_id);
    $domain_name = $domain['name'];
    $first_letter = strtoupper($domain_name[0]);
    $colors = PurviewHelper::generateLetterColors($first_letter);
    $domain_info = [
      'name' => $domain_name,
      'first_letter' => $first_letter,
      'background_color' => $colors['background'],
      'border_color' => $colors['border'],
    ];

    // Parent term.
    if (!empty($term['parentId'])) {
      $parent_term_id = $term['parentId'];
      $parent_term = $this->governanceDomainApiClient->getGlossaryTermMetadata($parent_term_id);
      $parent_link = Link::fromTextAndUrl($parent_term['name'], Url::fromUri("internal:/business-glossary/{$parent_term['id']}"));
      $parent_render_array = $parent_link->toRenderable();

      $parent_term_output = DeprecationHelper::backwardsCompatibleCall(
        currentVersion: \Drupal::VERSION,
        deprecatedVersion: '10.3',
        currentCallable: fn() => $this->renderer->renderInIsolation($parent_render_array),
        deprecatedCallable: fn() => $this->renderer->renderPlain($parent_render_array),
      );
    }
    else {
      $parent_term_output = '-';
    }

    // Acronyms.
    if (!empty($term['acronyms'])) {
      $acronyms = '';
      foreach ($term['acronyms'] as $acronym) {
        $acronyms .= $acronym . '<br>';
      }
    }
    else {
      $acronyms = '-';
    }

    // Custom attributes.
    $attributes = $this->classicTypesApiClient->getManagedAttributeDefinitions();
    $structured_attributes = [];
    if (!empty($term['managedAttributes'])) {
      $structured_attributes = $this->classicTypesApiClient->buildGroupedManagedAttributes(
        $term['managedAttributes'],
        $attributes
      );
    }

    // Associated data products.
    $related_entities = $this->governanceDomainApiClient->getRelatedEntityIdsForTerm($guid, 'DataProduct');
    $products = [];
    foreach ($related_entities as $entity_id) {
      $product = $this->governanceDomainApiClient->getDataProductMetadata($entity_id);
      $products[] = [
        'entity_id' => $entity_id,
        'name' => $product['name'],
      ];
    }

    // Canonical URL.
    $canonical_url = Url::fromRoute('purview_unified_catalog_ui.glossary_metadata_page', ['guid' => $guid], ['absolute' => TRUE])->toString();

    // Messages.
    $messages_build = [
      '#type' => 'status_messages',
    ];
    $messages_rendered = $this->renderer->render($messages_build);
    $messages = '<div data-drupal-messages>' . $messages_rendered . '</div>';

    return [
      '#attached' => [
        'library' => ['purview_unified_catalog_ui/unified_term_metadata_offcanvas'],
      ],
      'messages' => [
        '#markup' => $messages,
      ],
      'content' => [
        '#theme' => 'unified_catalog_term_metadata',
        '#data' => $term,
        '#name' => $title,
        '#status' => $status,
        '#description' => $description,
        '#domain' => $domain_info,
        '#owners' => $owner_output,
        '#authoring_info' => $authoring_info,
        '#parent_term' => $parent_term_output,
        '#acronyms' => $acronyms,
        '#attributes' => $structured_attributes,
        '#products' => $products,
        '#module_path' => $module_path,
        '#canonical_url' => $canonical_url,
      ],
    ];
  }

}
