<?php

namespace Drupal\purview_unified_catalog_ui\Controller;

use Drupal\Component\Datetime\Time;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Component\Utility\Html;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient;
use Drupal\drupal_purview\Utility\PurviewHelper;
use Drupal\purview_unified_catalog_ui\Form\UnifiedTermSearchForm;
use Drupal\purview_unified_catalog_ui\Utility\UnifiedCatalogHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for glossary term search and results.
 */
class UnifiedTermSearchController extends ControllerBase {

  /**
   * The governance domain API client.
   *
   * @var \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
   */
  protected PurviewGovernanceDomainApiClient $governanceDomainApiClient;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\Time
   */
  protected Time $time;

  /**
   * The Purview helper service.
   *
   * @var \Drupal\drupal_purview\Utility\PurviewHelper
   */
  protected PurviewHelper $purviewHelper;

  /**
   * The Purview Unified Catalog UI helper service.
   *
   * @var \Drupal\purview_unified_catalog_ui\Utility\UnifiedCatalogHelper
   */
  protected UnifiedCatalogHelper $unifiedCatalogHelper;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * The renderer service used to convert render arrays into HTML.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Module extension list (to resolve module paths).
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $moduleExtensionList;

  /**
   * Constructs the UnifiedTermSearchController.
   */
  public function __construct(
    FormBuilderInterface $formBuilder,
    PurviewGovernanceDomainApiClient $governanceDomainApiClient,
    ConfigFactoryInterface $configFactory,
    DateFormatterInterface $dateFormatter,
    Time $time,
    PurviewHelper $purviewHelper,
    CacheBackendInterface $cache,
    RendererInterface $renderer,
    RequestStack $requestStack,
    UnifiedCatalogHelper $unifiedCatalogHelper,
    ModuleExtensionList $moduleExtensionList,
  ) {
    $this->formBuilder = $formBuilder;
    $this->governanceDomainApiClient = $governanceDomainApiClient;
    $this->configFactory = $configFactory;
    $this->dateFormatter = $dateFormatter;
    $this->time = $time;
    $this->purviewHelper = $purviewHelper;
    $this->cache = $cache;
    $this->renderer = $renderer;
    $this->requestStack = $requestStack;
    $this->unifiedCatalogHelper = $unifiedCatalogHelper;
    $this->moduleExtensionList = $moduleExtensionList;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('form_builder'),
      $container->get('drupal_purview.governance_domain_api'),
      $container->get('config.factory'),
      $container->get('date.formatter'),
      $container->get('datetime.time'),
      $container->get('drupal_purview.helper'),
      $container->get('cache.default'),
      $container->get('renderer'),
      $container->get('request_stack'),
      $container->get('purview_unified_catalog_ui.helper'),
      $container->get('extension.list.module'),
    );
  }

  /**
   * Builds the main glossary search page with optional filters and view toggle.
   *
   * This method renders the glossary search form (if enabled in configuration),
   * with a toggle to switch between "List" and "Tree" views of glossary terms.
   * Search parameters (keywords, domain, owner) are extracted from the request,
   * and the appropriate view is rendered based on the 'view' query parameter.
   *
   * The result is a fully structured render array containing:
   * - The glossary search form (if enabled).
   * - AJAX-enabled toggle links to switch between list/tree views.
   * - A container for glossary term results (rendered in list or tree format).
   *
   * @return array
   *   A render array representing the glossary search page, including
   *   form, toggle links, and search results grouped by view type.
   */
  public function page(): array {
    // Get query strings.
    $request = $this->requestStack->getCurrentRequest();
    $keywords = Html::escape((string) $request->query->get('keywords'));
    $domain = Html::escape((string) ($request->query->get('domain') ?? 'all'));
    $attributes = Html::escape((string) ($request->query->get('attributes') ?? 'all'));
    $owner = Html::escape((string) ($request->query->get('owner') ?? 'all'));
    $view = Html::escape((string) ($request->query->get('view') ?? 'list'));

    // Get settings.
    $settings = $this->configFactory->get('purview_unified_catalog_ui.settings');
    $display_form = $settings->get('display_search_forms');

    $intro = $settings->get('glossary_intro');
    $intro_render = [
      '#type' => 'processed_text',
      '#text' => $intro['value'],
      '#format' => $intro['format'],
      '#prefix' => '<div class="glossary-intro">',
      '#suffix' => '</div>',
    ];

    if ($display_form) {
      return [
        '#type' => 'container',
        '#attributes' => ['class' => ['glossary-search-page']],
        'form_wrapper' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['glossary-search-form']],
          'form' => $this->formBuilder->getForm(UnifiedTermSearchForm::class),
        ],
        'intro' => $intro_render,
        'results_wrapper' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['glossary-search-results']],
          'toggle' => $this->buildViewToggle($view, $request),
          'results' => $view === 'tree'
            ? $this->buildTreeView($keywords, $domain, $owner, $attributes)
            : $this->buildListView($keywords, $domain, $owner, $attributes),
        ],
      ];
    }
    else {
      return [
        '#type' => 'container',
        '#attributes' => ['class' => ['glossary-search-page']],
        'intro' => $intro_render,
        'results_wrapper' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['glossary-search-results']],
          'toggle' => $this->buildViewToggle($view, $request),
          'results' => $view === 'tree'
            ? $this->buildTreeView($keywords, $domain, $owner, $attributes)
            : $this->buildListView($keywords, $domain, $owner, $attributes),
        ],
      ];
    }
  }

  /**
   * Builds the view mode toggle links for list and tree views.
   *
   * This method generates two AJAX-enabled links that allow users to toggle
   * between "List view" and "Tree view" for glossary terms. It inspects the
   * current active view mode to apply the 'is-active' class to the
   * corresponding toggle link for styling purposes.
   *
   * To ensure clean and reusable URLs, it removes internal AJAX-related query
   * parameters such as `_wrapper_format` and `_format` from the request before
   * generating the link URLs.
   *
   * The result is a render array suitable for inclusion inside a renderable
   * glossary search results wrapper, with all required AJAX libraries attached.
   *
   * @param string $view
   *   The current view mode, either 'list' or 'tree'.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request object used to extract query parameters.
   *
   * @return array
   *   A render array containing the list and tree view toggle links.
   */
  protected function buildViewToggle(string $view, $request): array {
    $query = $request->query->all();
    unset($query['_wrapper_format'], $query['_format']);

    // Get icons.
    $module_path = $this->moduleExtensionList->getPath('purview_unified_catalog_ui');
    $list_icon = file_get_contents($module_path . '/icons/list.svg');
    $tree_icon = file_get_contents($module_path . '/icons/tree-list.svg');

    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['glossary-view-toggle']],
      '#attached' => [
        'library' => [
          'core/drupal',
          'core/drupal.ajax',
        ],
      ],
      'list' => [
        '#type' => 'link',
        '#title' => [
          '#markup' => new FormattableMarkup('
            <span class="visually-hidden">@label</span>' . $list_icon,
            ['@label' => $this->t('List view')]
          ),
        ],
        '#url' => Url::fromRoute('purview_unified_catalog_ui.glossary_view_switch', ['view' => 'list'], [
          'query' => $query,
        ]),
        '#attributes' => [
          'class' => array_filter([
            'use-ajax',
            'view-switch',
            $view === 'list' ? 'is-active' : NULL,
          ]),
          'data-view' => 'list',
        ],
      ],
      'tree' => [
        '#type' => 'link',
        '#title' => [
          '#markup' => new FormattableMarkup('
            <span class="visually-hidden">@label</span>' . $tree_icon,
            ['@label' => $this->t('Tree view')]
          ),
        ],
        '#url' => Url::fromRoute('purview_unified_catalog_ui.glossary_view_switch', ['view' => 'tree'], [
          'query' => $query,
        ]),
        '#attributes' => [
          'class' => array_filter([
            'use-ajax',
            'view-switch',
            $view === 'tree' ? 'is-active' : NULL,
          ]),
          'data-view' => 'tree',
        ],
      ],
    ];
  }

  /**
   * Retrieves glossary terms from Purview with optional filtering and caching.
   *
   * This method queries the Microsoft Purview API for terms, supporting
   * optional filters for keyword, domain, and owner. If the domain is 'all',
   * it aggregates terms across available domains. To improve performance and
   * reduce API calls, results are cached per unique combination for one hour.
   *
   * Cache key is generated based on hash of the keyword, domain, and owner.
   * Cached data is tagged with 'purview_glossary_terms' to allow invalidation.
   *
   * @param string $keywords
   *   Optional keyword string to filter glossary terms by name or description.
   * @param string $domain
   *   The domain ID to filter terms by, or 'all' to include all domains.
   * @param mixed $owner
   *   Optional owner filter to restrict terms by associated owner(s).
   * @param mixed $attributes
   *   Optional attributes to filter terms against.
   *
   * @return array
   *   A flat array of glossary term data returned from the Purview API.
   */
  protected function getGlossaryTerms(string $keywords = '', string $domain = 'all', $owner = 'all', $attributes = 'all'): array {
    // Build cache key.
    $key_parts = [
      'purview_glossary_terms',
      'keywords:' . sha1($keywords),
      'domain:' . $domain,
      'owner:' . $owner,
      'attributes:' . sha1((string) $attributes),
    ];
    $cid = implode(':', $key_parts);

    $cache = $this->cache->get($cid);
    if ($cache && isset($cache->data)) {
      return $cache->data;
    }

    $results = [];

    if ($domain === 'all') {
      $all_domains = $this->governanceDomainApiClient->getGovernanceDomains();

      if (is_array($all_domains)) {
        foreach ($all_domains as $d) {
          $terms = $this->governanceDomainApiClient->getGlossaryTermsByDomain($d['id'], $keywords, $owner, $attributes);
          if (is_array($terms)) {
            $results = array_merge($results, $terms['value']);
          }
        }
      }
    }
    else {
      $terms = $this->governanceDomainApiClient->getGlossaryTermsByDomain($domain, $keywords, $owner, $attributes);
      $results = $terms['value'] ?? [];
    }

    // Cache result.
    $this->cache->set($cid, $results, strtotime('+1 hour'), ['purview_glossary_terms']);

    return $results;
  }

  /**
   * Builds the glossary term list view as an alphabetical render array.
   *
   * This method retrieves glossary terms based on optional keyword, domain, and
   * owner filters, then groups and sorts alphabetically by first letter
   * of the term name. Each group is rendered with a section heading, a row of
   * column labels, and one or more glossary term rows using the
   * 'unified_catalog_term_result' theme.
   *
   * The result includes an anchor navigation block to jump to each lettered
   * section, a result summary, and attached JS/CSS libraries to support
   * AJAX and UI behaviors.
   *
   * @param string|null $keywords
   *   Optional search keywords to filter glossary terms.
   * @param string $domains
   *   A domain ID to filter results by, or 'all' to include all domains.
   * @param mixed $owner
   *   Optional owner filter to restrict results to a specific user or group.
   * @param mixed $attributes
   *   Optional attributes to filter terms against.
   *
   * @return array
   *   A renderable array representing the grouped list view of glossary terms,
   *   including anchors, term rows, and summary text.
   */
  protected function buildListView(?string $keywords, string $domains, $owner, $attributes): array {
    // Get results.
    $keywords = $keywords ?? '';
    $results = $this->getGlossaryTerms($keywords, $domains, $owner, $attributes);
    $settings = $this->configFactory->get('purview_unified_catalog_ui.settings');

    if (empty($results)) {
      return [
        '#type' => 'container',
        '#attributes' => ['class' => ['empty-results']],
        '#markup' => $this->t('No results found.'),
        '#attached' => [
          'library' => [
            'purview_unified_catalog_ui/unified_term_search',
          ],
        ],
      ];
    }

    // Group by first letter.
    $grouped = [];
    foreach ($results as $term) {
      $name = $term['name'] ?? '';
      $firstChar = strtoupper(mb_substr($name, 0, 1));

      if (ctype_alpha($firstChar)) {
        $key = $firstChar;
      }
      elseif (ctype_digit($firstChar)) {
        $key = '1';
      }
      elseif ($firstChar !== '') {
        $key = '#';
      }
      else {
        continue;
      }

      $grouped[$key][] = $term;
    }

    // Sort groups and terms inside each group.
    ksort($grouped);
    foreach ($grouped as &$terms) {
      usort($terms, fn($a, $b) => strcasecmp($a['name'], $b['name']));
    }
    unset($terms);

    // Build anchor links.
    $all_sections = array_merge(['#', '1'], range('A', 'Z'));
    $output = [
      '#type' => 'container',
      '#attached' => [
        'library' => [
          'core/drupal.dialog.ajax',
          'purview_unified_catalog_ui/unified_term_search',
          'purview_unified_catalog_ui/clear_search',
        ],
      ],
      'summary' => [
        '#markup' => $this->formatPlural(count($results), '@count result found.', '@count results found.'),
        '#prefix' => '<div class="glossary-search-summary">',
        '#suffix' => '</div>',
      ],
      'anchors' => [
        '#prefix' => '<section class="glossary-anchor-links"><div class="nav-wrapper"><h2 class="visually-hidden">' . $this->t('Glossary term navigation') . '</h2> ',
        '#suffix' => '</div></section>',
        '#markup' => implode(' ', array_map(function ($char) use ($grouped) {
          return isset($grouped[$char])
            ? "<a href=\"#letter-{$char}\">{$char}</a>"
            : "<span class=\"glossary-anchor-disabled\">{$char}</span>";
        }, $all_sections)),
      ],
    ];

    // Get domain list.
    $domain_lookup = [];
    foreach ($this->governanceDomainApiClient->getGovernanceDomains() as $d) {
      $domain_lookup[$d['id']] = $d['name'];
    }

    // Build grouped items.
    foreach ($grouped as $char => $terms) {
      $section_id = "letter-$char";
      $output[$section_id . '_heading'] = [
        '#markup' => "<div class='term-section' id=\"$section_id\">$char</div>",
      ];

      $output[$section_id . '_prefix'] = [
        '#markup' => "<div class=\"section-wrapper section-$char\">",
      ];

      $attribute_label_raw = (string) ($settings->get('attribute_filter_label') ?? '');
      $attribute_label = $attribute_label_raw !== ''
        ? Html::escape($attribute_label_raw)
        : $this->t('- Attribute -');

      $output[$section_id . '_row_headings'] = [
        '#markup' => '<div class="row-header">
                        <div class="column">' . $this->t('Glossary term name') . '</div>
                        <div class="column">' . $this->t('Description') . '</div>
                        <div class="column">' . $this->t('Governance domain') . '</div>
                        <div class="column">' . $this->t('Contact') . '</div>
                        <div class="column">' . $attribute_label . '</div>
                      </div>',
      ];

      foreach ($terms as $term) {
        // Get term render data.
        $render_data = $this->prepareTermRenderData($term, $domain_lookup);

        $output["{$section_id}_item_" . $term['id']] = [
          '#theme' => 'unified_catalog_term_result',
          '#link' => $render_data['link'],
          '#term' => $term,
          '#name' => $term['name'],
          '#status' => $term['status'] ?? '',
          '#description' => $render_data['description'],
          '#desc_truncated' => $render_data['desc_truncated'],
          '#domain' => $render_data['domain'],
          '#updated' => $render_data['updated'],
          '#owner' => $render_data['owner'],
          '#attributes_text' => $render_data['attributes_text'],
        ];
      }

      $output[$section_id . '_suffix'] = [
        '#markup' => "</div>",
      ];
    }

    return $output;
  }

  /**
   * Builds the glossary term tree view as a hierarchical render array.
   *
   * This method retrieves glossary terms based on optional keyword, domain, and
   * owner filters, then organizes them into a parent-child tree structure using
   * the `parentId` attribute on each term. The resulting hierarchy is rendered
   * recursively using renderTermRecursive(), wrapped in a container suitable
   * for AJAX replacement on the frontend.
   *
   * The top-level terms are sorted alphabetically (A–Z), and each term includes
   * metadata like domain color, truncated description, last updated date.
   *
   * @param string|null $keywords
   *   Optional search keywords to filter glossary terms.
   * @param string $domains
   *   A domain ID to filter results by, or 'all' to include all domains.
   * @param mixed $owner
   *   Optional owner filter to restrict results to a specific user or group.
   * @param mixed $attributes
   *   Optional attributes to filter terms against.
   *
   * @return array
   *   A renderable array representing the tree view of glossary terms,
   *   including summary text, attached libraries, and all hierarchical nodes.
   */
  protected function buildTreeView(?string $keywords, string $domains, $owner, $attributes): array {
    // Get results.
    $keywords = $keywords ?? '';
    $results = $this->getGlossaryTerms($keywords, $domains, $owner, $attributes);

    if (empty($results)) {
      return [
        '#type' => 'container',
        '#attributes' => ['class' => ['empty-results']],
        '#markup' => $this->t('No results found.'),
        '#attached' => [
          'library' => [
            'purview_unified_catalog_ui/unified_term_search',
          ],
        ],
      ];
    }

    // Get domain info once.
    $domain_lookup = [];
    foreach ($this->governanceDomainApiClient->getGovernanceDomains() as $d) {
      $domain_lookup[$d['id']] = $d['name'];
    }

    // Index by ID and track missing parent IDs.
    $terms_by_id = [];
    $missing_parents = [];

    foreach ($results as $term) {
      $term['children'] = [];
      $terms_by_id[$term['id']] = $term;
      $parent_id = $term['parentId'] ?? NULL;
      if ($parent_id && !isset($terms_by_id[$parent_id])) {
        $missing_parents[$parent_id] = TRUE;
      }
    }

    // Load missing parents recursively.
    while (!empty($missing_parents)) {
      $next_batch = [];
      foreach (array_keys($missing_parents) as $parent_id) {
        $parent_term = $this->governanceDomainApiClient->getGlossaryTermMetadata($parent_id);
        if ($parent_term && is_array($parent_term)) {
          $parent_term['children'] = [];
          $terms_by_id[$parent_id] = $parent_term;
          $grand_parent_id = $parent_term['parentId'] ?? NULL;
          if ($grand_parent_id && !isset($terms_by_id[$grand_parent_id])) {
            $next_batch[$grand_parent_id] = TRUE;
          }
        }
      }
      $missing_parents = $next_batch;
    }

    // Build full hierarchy.
    $tree = [];
    foreach ($terms_by_id as $id => &$term) {
      $parent_id = $term['parentId'] ?? NULL;
      if ($parent_id && isset($terms_by_id[$parent_id])) {
        $terms_by_id[$parent_id]['children'][] = &$term;
      }
      else {
        $tree[] = &$term;
      }
    }
    unset($term);

    // Sort top-level terms A-Z.
    usort($tree, fn($a, $b) => strcasecmp($a['name'], $b['name']));

    // Render recursively.
    $output = [
      '#type' => 'container',
      '#attributes' => ['class' => ['tree-view']],
      '#attached' => [
        'library' => [
          'core/drupal.dialog.ajax',
          'purview_unified_catalog_ui/unified_term_search',
        ],
      ],
      'summary' => [
        '#markup' => $this->formatPlural(count($results), '@count result found.', '@count results found.'),
        '#prefix' => '<div class="glossary-search-summary">',
        '#suffix' => '</div>',
      ],
    ];

    // Attribute label.
    $settings = $this->configFactory->get('purview_unified_catalog_ui.settings');
    $attribute_label_raw = (string) ($settings->get('attribute_filter_label') ?? '');
    $attribute_label = $attribute_label_raw !== ''
      ? Html::escape($attribute_label_raw)
      : $this->t('- Attribute -');

    $output['row_headings'] = [
      '#markup' => '<div class="row-header">
                      <div class="column">' . $this->t('Glossary term name') . '</div>
                      <div class="column">' . $this->t('Description') . '</div>
                      <div class="column">' . $this->t('Governance domain') . '</div>
                      <div class="column">' . $this->t('Contact') . '</div>
                      <div class="column">' . $attribute_label . '</div>
                    </div>',
    ];

    $output['items'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['tree-items']],
    ];

    foreach ($tree as $term) {
      $output['items'][] = $this->renderTermRecursive($term, $domain_lookup);
    }

    return $output;
  }

  /**
   * Recursively renders a glossary term and its children for the tree view.
   *
   * This method builds a render array for a glossary term using the
   * 'unified_catalog_term_result' theme and nests any child terms within
   * a container for hierarchical display. It is used by the tree view to
   * display glossary terms in a collapsible or indented structure.
   *
   * The method relies on prepareTermRenderData() to generate consistent
   * output for term display elements such as links, domain badges, dates,
   * and truncated descriptions.
   *
   * @param array $term
   *   The glossary term data, including any preloaded 'children' sub-terms.
   * @param array $domain_lookup
   *   An associative array mapping domain IDs to domain names, used to
   *   populate domain-related display data.
   *
   * @return array
   *   A renderable array representing the glossary term and its children,
   *   suitable for use in the tree view render pipeline.
   */
  protected function renderTermRecursive(array $term, array $domain_lookup): array {
    // Get term render data.
    $render_data = $this->prepareTermRenderData($term, $domain_lookup);

    // Wrap the theme output and children together.
    $render = [
      '#type' => 'container',
      '#attributes' => ['class' => ['tree-term-wrapper']],
      'term' => [
        '#theme' => 'unified_catalog_term_result',
        '#link' => $render_data['link'],
        '#term' => $term,
        '#name' => $term['name'],
        '#status' => $term['status'] ?? '',
        '#description' => $render_data['description'],
        '#desc_truncated' => $render_data['desc_truncated'],
        '#domain' => $render_data['domain'],
        '#updated' => $render_data['updated'],
        '#owner' => $render_data['owner'],
        '#attributes_text' => $render_data['attributes_text'],
      ],
    ];

    // Add children inside this wrapper.
    if (!empty($term['children'])) {
      $render['children'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['tree-children']],
      ];
      foreach ($term['children'] as $child) {
        $render['children'][] = $this->renderTermRecursive($child, $domain_lookup);
      }
    }

    return $render;
  }

  /**
   * Prepares common render data for a glossary term.
   *
   * This method builds a render array for the term link, formats the domain
   * metadata (including generated colors), formats the last modified timestamp,
   * and sanitizes and truncates the term description.
   *
   * The returned array is intended to be used in both list and tree views to
   * standardize how terms are rendered using unified_catalog_term_result
   * theme template.
   *
   * @param array $term
   *   The glossary term data as returned by the Purview API.
   * @param array $domain_lookup
   *   An associative array of domain IDs to domain names.
   *
   * @return array
   *   An array containing:
   *   - link: Render array for the offcanvas term link.
   *   - domain: Associative array with domain name and styling info.
   *   - updated: Formatted last modified date string.
   *   - description: Sanitized plain-text version of the full description.
   *   - desc_truncated: Truncated version of the description (max 120 chars).
   *   - owner: The owner of the glossary term.
   */
  protected function prepareTermRenderData(array $term, array $domain_lookup): array {
    $link = [
      '#type' => 'link',
      '#title' => $term['name'],
      '#url' => Url::fromRoute('purview_unified_catalog_ui.glossary_metadata_offcanvas', ['guid' => $term['id']]),
      '#options' => [
        'attributes' => ['class' => ['use-ajax', 'glossary-term-link']],
      ],
    ];

    // Domain.
    $domain_name = $domain_lookup[$term['domain']] ?? '';
    $first_letter = strtoupper($domain_name[0] ?? '?');
    $colors = PurviewHelper::generateLetterColors($first_letter);
    $domain_info = [
      "name" => $domain_name,
      "first_letter" => $first_letter,
      "background_color" => $colors['background'],
      "border_color" => $colors['border'],
    ];

    // Updated date.
    $modified_date = $term['systemData']['lastModifiedAt'] ?? NULL;
    $updated_text = $modified_date
      ? $this->dateFormatter->format((new \DateTime($modified_date))->getTimestamp(), 'custom', 'n/j/Y, g:i A')
      : $this->t('Updated date unavailable');

    // Description.
    $description = preg_replace('/<\/(p|div|br|h[1-6]|li|ul|ol|table|tr|td)>/i', ' </$1>', $term['description']);
    $description = Html::decodeEntities(strip_tags($description));
    $truncated = mb_strlen($description) > 120 ? mb_substr($description, 0, 120) . '...' : $description;

    // Owners.
    $owners = $this->purviewHelper->renderOwnersHorizontal($term['contacts']['owner']);

    // Attributes.
    $attributes = $this->unifiedCatalogHelper->extractManagedAttributeValues($term);
    $attribute_text = !empty($attributes) ? implode(', ', $attributes) : '--';

    return [
      'link' => $link,
      'domain' => $domain_info,
      'updated' => $updated_text,
      'description' => $description,
      'desc_truncated' => $truncated,
      'owner' => $owners,
      'attributes_text' => $attribute_text,
    ];
  }

  /**
   * AJAX callback to switch between list and tree view modes.
   *
   * Retrieves the current search parameters (keywords, domain, owner)
   * from the query string, then renders either the list or tree view based on
   * the provided $view argument. The resulting output replaces the contents of
   * the `.glossary-search-results` container on the page via an AjaxResponse.
   *
   * @param string $view
   *   The view mode to render, either 'list' or 'tree'.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response containing the updated glossary view HTML.
   */
  public function ajaxSwitchView(string $view): AjaxResponse {
    $request = $this->requestStack->getCurrentRequest();
    $keywords = Html::escape((string) $request->query->get('keywords'));
    $domain = Html::escape((string) ($request->query->get('domain') ?? 'all'));
    $owner = Html::escape((string) ($request->query->get('owner') ?? 'all'));
    $attributes = Html::escape((string) ($request->query->get('attributes') ?? 'all'));

    $wrapper = [
      '#type' => 'container',
      '#attached' => [
        'library' => [
          'core/drupal',
          'core/drupal.ajax',
          'purview_unified_catalog_ui/unified_term_search',
        ],
      ],
      'toggle' => $this->buildViewToggle($view, $request),
      'results' => $view === 'tree'
        ? $this->buildTreeView($keywords, $domain, $owner, $attributes)
        : $this->buildListView($keywords, $domain, $owner, $attributes),
    ];

    $rendered = $this->renderer->renderRoot($wrapper);

    $response = new AjaxResponse();
    $response->addCommand(new HtmlCommand('.glossary-search-results', $rendered));
    return $response;
  }

}
