<?php

namespace Drupal\purview_unified_catalog_ui\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\drupal_purview\Service\GraphApiClient;
use Drupal\drupal_purview\Service\PurviewClassicTypesApiClient;
use Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides the unified term search form.
 */
class UnifiedTermSearchForm extends FormBase {

  /**
   * The Classic types API client.
   *
   * @var \Drupal\drupal_purview\Service\PurviewClassicTypesApiClient
   */
  protected PurviewClassicTypesApiClient $classicTypesApiClient;

  /**
   * The Purview governance domain API client.
   *
   * @var \Drupal\drupal_purview\Service\PurviewGovernanceDomainApiClient
   */
  protected PurviewGovernanceDomainApiClient $apiClient;

  /**
   * The Graph API client.
   *
   * @var \Drupal\drupal_purview\Service\GraphApiClient
   */
  protected GraphApiClient $graphApiClient;

  /**
   * Constructs a new UnifiedTermSearchForm.
   */
  public function __construct(
    PurviewClassicTypesApiClient $classicTypesApiClient,
    PurviewGovernanceDomainApiClient $apiClient,
    GraphApiClient $graphApiClient,
    RequestStack $requestStack,
  ) {
    $this->classicTypesApiClient = $classicTypesApiClient;
    $this->apiClient = $apiClient;
    $this->graphApiClient = $graphApiClient;
    $this->requestStack = $requestStack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('drupal_purview.api_client'),
      $container->get('drupal_purview.governance_domain_api'),
      $container->get('drupal_purview.graph_api_client'),
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'unified_term_search_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('purview_unified_catalog_ui.settings');

    $request = $this->requestStack->getCurrentRequest();
    $query_keywords = $request->query->get('keywords', '');

    // Build Owner options from the API.
    if ($config->get('filter_owner')) {
      $query_owner = $request->query->get('owner', 'all');
      $facets = $this->apiClient->getFacetValues('terms', ['owner']);
      $owner_options = [
        'all' => $this->t('- Owner -'),
      ];
      foreach ($facets['owner'] as $owner_data) {
        $owner = $this->graphApiClient->getUserObjectFromGuid($owner_data['value']);
        $owner_options[$owner['id']] = $owner['displayName'];
      }
      asort($owner_options, SORT_NATURAL | SORT_FLAG_CASE);
    }

    // Build Domain options from the API.
    if ($config->get('filter_domain')) {
      $query_domain = $request->query->get('domain', 'all');
      $domains = $this->apiClient->getGovernanceDomains();
      $domain_options = ['all' => $this->t('- Domain -')];

      if (is_array($domains)) {
        $tree = $this->apiClient->buildGovernanceDomainHierarchy($domains);
        $this->apiClient->formatHierarchySelectList($tree, $domain_options);
      }
    }

    // Build custom attribute options.
    if ($config->get('filter_attributes')) {
      $query_attribute = $request->query->get('attributes', 'all');
      $attribute_filter_label = $config->get('attribute_filter_label') ?: $this->t('- Attribute -');
      $label = '- ' . $attribute_filter_label . ' -';
      $attribute_options = ['all' => $label];
      $attributes = $this->classicTypesApiClient->getManagedAttributeDefinitions();

      if (is_array($attributes)) {
        // Normalize filter to an array of "group|field" values.
        $filter_raw = trim((string) ($config->get('attribute_filter') ?? ''));
        $allowed = [];

        if ($filter_raw !== '') {
          foreach (preg_split('/\R/', $filter_raw) as $line) {
            $line = trim($line);
            if ($line !== '') {
              $allowed[$line] = TRUE;
            }
          }
        }

        foreach ($attributes as $attribute) {
          $group = $attribute['group'] ?? '';
          $field = $attribute['field'] ?? '';
          $pair  = "$group|$field";

          $disabled   = $attribute['is_disabled'] ?? FALSE;
          $is_enabled = ($disabled === FALSE || $disabled === 'false');

          // Add if enabled AND (no filter OR matches filter).
          if ($is_enabled && !empty($attribute['options'])
              && (empty($allowed) || isset($allowed[$pair]))) {
            foreach ($attribute['options'] as $option) {
              $attribute_options["$group.$field|$option"] = $option;
            }
          }
        }
      }
    }

    // Text search and submit button.
    $form['search_group'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['search-group']],
    ];

    $form['search_group']['keywords'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search enterprise glossary terms'),
      '#title_display' => 'invisible',
      '#default_value' => $query_keywords,
      '#attributes' => [
        'placeholder' => $this->t('Enter keywords...'),
      ],
    ];

    if ($config->get('filter_domain')) {
      $form['search_group']['domains'] = [
        '#type' => 'select',
        '#title' => t('Filter by domain'),
        '#title_display' => 'invisible',
        '#options' => $domain_options,
        '#default_value' => $query_domain,
      ];
    }

    if ($config->get('filter_owner')) {
      $form['search_group']['owner'] = [
        '#type' => 'select',
        '#title' => t('Filter by owner'),
        '#title_display' => 'invisible',
        '#options' => $owner_options,
        '#default_value' => $query_owner,
      ];
    }

    if ($config->get('filter_attributes')) {
      $form['search_group']['attributes'] = [
        '#type' => 'select',
        '#title' => t('Filter by attribute'),
        '#title_display' => 'invisible',
        '#options' => $attribute_options,
        '#default_value' => $query_attribute,
      ];
    }

    $form['search_group']['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Apply'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Redirect to the search page with query parameters.
    $config = $this->config('purview_unified_catalog_ui.settings');
    $query = ['keywords' => $form_state->getValue('keywords')];

    if ($config->get('filter_domain')) {
      $query['domain'] = $form_state->getValue('domains');
    }

    if ($config->get('filter_owner')) {
      $query['owner'] = $form_state->getValue('owner');
    }

    if ($config->get('filter_attributes')) {
      $query['attributes'] = $form_state->getValue('attributes');
    }

    $form_state->setRedirect(
      'purview_unified_catalog_ui.glossary_search',
      [],
      ['query' => $query]
    );
  }

}
