<?php

namespace Drupal\drupal_purview\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Config form for Drupal Purview integration settings.
 *
 * This form lets administrators configure the Microsoft Purview and
 * Microsoft Graph API credentials and endpoints used by the
 * drupal_purview module. Values are stored in the
 * "drupal_purview.settings" configuration object.
 */
class PurviewSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['drupal_purview.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'drupal_purview_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('drupal_purview.settings');

    // Purview API.
    $form['purview'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Microsoft Purview API'),
    ];

    $form['purview']['generic'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Generic Purview settings'),
    ];

    $form['purview']['generic']['purview_account_guid'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Purview Account GUID'),
      '#default_value' => $config->get('purview_account_guid'),
      '#description' => $this->t('The Purview account GUID used to build authentication and API URLs for accessing Purview resources.'),
      '#required' => TRUE,
    ];

    $form['purview']['generic']['client_id'] = [
      '#title' => $this->t('Client ID'),
      '#type' => 'key_select',
      '#default_value' => $config->get('client_id'),
      '#description' => $this->t('Select a key that holds an Active Directory service principal client ID. This client ID will be used when requesting an access token to authorize the connect to the Purview REST API.'),
      '#required' => TRUE,
    ];

    $form['purview']['generic']['client_secret'] = [
      '#title' => $this->t('Client secret'),
      '#type' => 'key_select',
      '#default_value' => $config->get('client_secret'),
      '#description' => $this->t('Select a key that holds an Active Directory service principal client secret. This secret will be used along with the client ID when requesting an access token to authorize the connect to the Purview REST API.'),
      '#required' => TRUE,
    ];

    $form['purview']['unified'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Unified catalog settings'),
    ];

    $form['purview']['unified']['api_version'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API version'),
      '#default_value' => $config->get('api_version'),
      '#description' => $this->t('The API version to use for Classic types. Example: 2024-02-01-preview'),
      '#required' => TRUE,
    ];

    $form['purview']['classic'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Classic types settings'),
    ];

    $form['purview']['classic']['rest_endpoint'] = [
      '#type' => 'url',
      '#title' => $this->t('Purview REST API Endpoint'),
      '#default_value' => $config->get('rest_endpoint'),
      '#description' => $this->t('Enter the Purview REST API base url to use. Example: https://api.purview-service.microsoft.com'),
      '#required' => TRUE,
    ];

    // Graph API.
    $form['graph_api'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Microsoft Graph API'),
    ];

    $form['graph_api']['graph_tenant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Graph Tenant ID'),
      '#default_value' => $config->get('graph_tenant_id'),
      '#required' => TRUE,
    ];

    $form['graph_api']['graph_client_id'] = [
      '#title' => $this->t('Graph Client ID'),
      '#type' => 'key_select',
      '#default_value' => $config->get('graph_client_id'),
      '#required' => TRUE,
    ];

    $form['graph_api']['graph_client_secret'] = [
      '#title' => $this->t('Graph Client Secret'),
      '#type' => 'key_select',
      '#default_value' => $config->get('graph_client_secret'),
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    $this->config($this->getEditableConfigNames()[0])
      ->set('purview_account_guid', $form_state->getValue('purview_account_guid'))
      ->set('rest_endpoint', $form_state->getValue('rest_endpoint'))
      ->set('client_id', $form_state->getValue('client_id'))
      ->set('client_secret', $form_state->getValue('client_secret'))
      ->set('api_version', $form_state->getValue('api_version'))
      ->set('graph_tenant_id', $form_state->getValue('graph_tenant_id'))
      ->set('graph_client_id', $form_state->getValue('graph_client_id'))
      ->set('graph_client_secret', $form_state->getValue('graph_client_secret'))
      ->save();
  }

}
