<?php

namespace Drupal\drupal_purview\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\key\KeyRepository;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Psr\Log\LoggerInterface;

/**
 * Service for accessing Microsoft Graph API user data.
 */
class GraphApiClient {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The HTTP client used for API requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * Logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Key repository service for retrieving credentials.
   *
   * @var \Drupal\key\KeyRepository
   */
  protected KeyRepository $keyRepository;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * Cached Microsoft Graph access token.
   *
   * @var string|null
   */
  protected ?string $accessToken = NULL;

  /**
   * Constructs the GraphApiClient service.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param \Drupal\key\KeyRepository $keyRepository
   *   The key repository.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache service.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    ClientInterface $httpClient,
    LoggerChannelFactoryInterface $loggerFactory,
    KeyRepository $keyRepository,
    CacheBackendInterface $cache,
  ) {
    $this->configFactory = $configFactory;
    $this->httpClient = $httpClient;
    $this->logger = $loggerFactory->get('drupal_purview');
    $this->keyRepository = $keyRepository;
    $this->cache = $cache;
  }

  /**
   * Retrieves the full user object from Azure AD by GUID.
   *
   * @param string $guid
   *   The Azure AD user GUID.
   *
   * @return array|null
   *   The user object as an associative array, or NULL on failure.
   */
  public function getUserObjectFromGuid(string $guid): ?array {
    $cid = 'graph_user:' . $guid;

    // Try to load from cache first.
    if ($cache = $this->cache->get($cid)) {
      return $cache->data;
    }

    if (!$this->authenticate()) {
      return NULL;
    }

    try {
      $response = $this->httpClient->request('GET', "https://graph.microsoft.com/v1.0/users/{$guid}", [
        'headers' => [
          'Authorization' => "Bearer {$this->accessToken}",
          'Accept' => 'application/json',
        ],
      ]);

      $user = json_decode($response->getBody()->getContents(), TRUE);

      // Cache for 6 hours.
      if ($user) {
        $this->cache->set($cid, $user, strtotime('+1 day'), ['graph_user']);
      }

      return $user;
    }
    catch (RequestException $e) {
      $this->logger->warning('Graph API request failed for GUID @guid: @error', [
        '@guid' => $guid,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Resolves an Azure AD user GUID to a display name using Microsoft Graph.
   *
   * @param string $guid
   *   The Azure AD user GUID.
   *
   * @return string|null
   *   The display name or mail, or NULL if not found.
   */
  public function resolveUserNameByGuid(string $guid): ?string {
    $user = $this->getUserObjectFromGuid($guid);
    return $user['displayName'] ?? $user['mail'] ?? NULL;
  }

  /**
   * Authenticates with Microsoft Graph using client credentials flow.
   *
   * @return bool
   *   TRUE if authentication succeeded, FALSE otherwise.
   */
  protected function authenticate(): bool {
    if ($this->accessToken) {
      return TRUE;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $clientId = $this->keyRepository->getKey($config->get('graph_client_id'))->getKeyValue();
    $clientSecret = $this->keyRepository->getKey($config->get('graph_client_secret'))->getKeyValue();
    $tenantId = $config->get('graph_tenant_id');
    $url = "https://login.microsoftonline.com/{$tenantId}/oauth2/v2.0/token";

    try {
      $response = $this->httpClient->request('POST', $url, [
        'form_params' => [
          'grant_type' => 'client_credentials',
          'client_id' => $clientId,
          'client_secret' => $clientSecret,
          'scope' => 'https://graph.microsoft.com/.default',
        ],
        'headers' => [
          'Content-Type' => 'application/x-www-form-urlencoded',
          'Accept' => 'application/json',
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);
      $this->accessToken = $data['access_token'] ?? NULL;

      return !empty($this->accessToken);
    }
    catch (RequestException $e) {
      $this->logger->error('Graph API authentication failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

}
