<?php

namespace Drupal\drupal_purview\Service;

use GuzzleHttp\ClientInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Psr\Log\LoggerInterface;
use Drupal\key\KeyRepository;

/**
 * Provides a service to authenticate with Microsoft Purview using credentials.
 */
class PurviewAuthenticationService {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The HTTP client for making requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The key repository service from the Key module.
   *
   * @var \Drupal\key\KeyRepository
   */
  protected KeyRepository $keyRepository;

  /**
   * The access token retrieved from Microsoft Purview.
   *
   * @var string|null
   */
  protected ?string $accessToken = NULL;

  /**
   * Constructs the PurviewAuthenticationService.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param \Drupal\key\KeyRepository $keyRepository
   *   The key repository.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    ClientInterface $httpClient,
    LoggerChannelFactoryInterface $loggerFactory,
    KeyRepository $keyRepository,
  ) {
    $this->configFactory = $configFactory;
    $this->httpClient = $httpClient;
    $this->logger = $loggerFactory->get('drupal_purview');
    $this->keyRepository = $keyRepository;
  }

  /**
   * Retrieves the Microsoft Purview access token.
   *
   * If the token has already been retrieved, the cached version is returned.
   *
   * @return string|null
   *   The access token if available, or NULL on failure.
   */
  public function getAccessToken(): ?string {
    if ($this->accessToken) {
      return $this->accessToken;
    }

    $config = $this->configFactory->get('drupal_purview.settings');

    try {
      $clientId = $this->keyRepository->getKey($config->get('client_id'))->getKeyValue();
      $clientSecret = $this->keyRepository->getKey($config->get('client_secret'))->getKeyValue();
    }
    catch (\Throwable $e) {
      $this->logger->error('Key retrieval failed: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }

    $tenant_id = $config->get('tenant_id');
    $auth_url = "https://login.microsoftonline.com/{$tenant_id}/oauth2/v2.0/token";

    try {
      $response = $this->httpClient->request('POST', $auth_url, [
        'form_params' => [
          'grant_type' => 'client_credentials',
          'client_id' => $clientId,
          'client_secret' => $clientSecret,
          'scope' => 'https://purview.azure.net/.default',
        ],
        'headers' => [
          'Content-Type' => 'application/x-www-form-urlencoded',
          'Accept' => 'application/json',
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);
      $this->accessToken = $data['access_token'] ?? NULL;
      return $this->accessToken;
    }
    catch (\Exception $e) {
      $this->logger->error('Purview authentication failed: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }
  }

}
