<?php

namespace Drupal\drupal_purview\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Psr\Log\LoggerInterface;

/**
 * Provides a service for accessing Microsoft Purview Classic Types API.
 */
class PurviewClassicTypesApiClient {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The Guzzle HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The Purview authentication service.
   *
   * @var \Drupal\drupal_purview\Service\PurviewAuthenticationService
   */
  protected PurviewAuthenticationService $authService;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * Constructs a new PurviewClassicTypesApiClient object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger channel factory.
   * @param \Drupal\drupal_purview\Service\PurviewAuthenticationService $authService
   *   The Purview authentication service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache service.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    ClientInterface $httpClient,
    LoggerChannelFactoryInterface $loggerFactory,
    PurviewAuthenticationService $authService,
    CacheBackendInterface $cache,
  ) {
    $this->configFactory = $configFactory;
    $this->httpClient = $httpClient;
    $this->logger = $loggerFactory->get('drupal_purview');
    $this->authService = $authService;
    $this->cache = $cache;
  }

  /**
   * Retrieves the list of glossaries from Microsoft Purview.
   *
   * @return array|null
   *   An array of glossaries, or NULL on failure.
   */
  public function getGlossaries(): ?array {
    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $baseUrl = rtrim($config->get('rest_endpoint'), '/');
    $url = "{$baseUrl}/catalog/api/atlas/v2/glossary";

    try {
      $response = $this->httpClient->get($url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
        ],
      ]);
      $body = $response->getBody()->getContents();
      return json_decode($body, TRUE);
    }
    catch (RequestException $e) {
      $this->logger->error('Failed to retrieve glossaries: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Searches glossary terms from Microsoft Purview.
   *
   * @param string $keywords
   *   The search keywords. Default is an empty string for wildcard search.
   * @param int $limit
   *   Number of results per page. Defaults to 50 (maximum from Purview).
   * @param string $glossary
   *   The glossary name to filter by. Defaults to 'Glossary'.
   * @param int|null $offset
   *   The result offset for pagination. If NULL, performs paging automatically.
   *
   * @return array|null
   *   An array of glossary term search results, or NULL on failure.
   */
  public function searchGlossaryTerms(string $keywords = '', int $limit = 50, string $glossary = 'Glossary', ?int $offset = NULL): ?array {
    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $baseUrl = rtrim($config->get('rest_endpoint'), '/');
    $url = "{$baseUrl}/datamap/api/search/query";

    $filter = [
      ['objectType' => 'Glossary terms'],
      ['entityType' => 'AtlasGlossaryTerm'],
    ];

    if (strtolower($glossary) !== 'all') {
      $filter[] = ['glossary' => $glossary];
    }

    if (is_null($offset)) {
      $offset = 0;
    }

    $results = [];

    try {
      while (TRUE) {
        $data = [
          'keywords' => $keywords,
          'limit' => $limit,
          'offset' => $offset,
          'filter' => ['and' => $filter],
        ];

        $response = $this->httpClient->post($url, [
          'body' => json_encode($data),
          'headers' => [
            'Authorization' => "Bearer {$accessToken}",
            'Content-Type' => 'application/json',
          ],
        ]);

        $body = json_decode($response->getBody()->getContents(), TRUE);

        if (empty($body['value'])) {
          break;
        }

        $results = array_merge($results, $body['value']);

        // Stop if fewer than $limit results returned (end of paging).
        if (count($body['value']) < $limit) {
          break;
        }

        $offset += $limit;
      }

      return $results;
    }
    catch (RequestException $e) {
      $this->logger->error('Purview API request failed: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Retrieves detailed metadata for a glossary term by GUID.
   *
   * @param string $guid
   *   The unique identifier (GUID) of the glossary term.
   *
   * @return array|null
   *   An associative array of glossary term metadata, or NULL on failure.
   */
  public function getTermMetadata(string $guid): ?array {
    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $baseUrl = rtrim($config->get('rest_endpoint'), '/');
    $url = "{$baseUrl}/catalog/api/atlas/v2/entity/guid/{$guid}";

    try {
      $response = $this->httpClient->get($url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
        ],
      ]);
      $body = $response->getBody()->getContents();
      return json_decode($body, TRUE);
    }
    catch (RequestException $e) {
      $this->logger->error('Failed to fetch term metadata for @guid: @error', [
        '@guid' => $guid,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets metadata definitions for all managed attributes (custom metadata).
   *
   * @param bool $bypass_cache
   *   Whether to bypass the cache and fetch fresh data.
   *
   * @return array|null
   *   An associative array of managed attributes, keyed by "Group.Field".
   */
  public function getManagedAttributeDefinitions(bool $bypass_cache = FALSE): ?array {
    $cid = 'purview_managed_attribute_definitions';

    if (!$bypass_cache) {
      $cache = $this->cache->get($cid);
      if ($cache && isset($cache->data)) {
        return $cache->data;
      }
    }

    $accessToken = $this->authService->getAccessToken();
    if (!$accessToken) {
      return NULL;
    }

    $config = $this->configFactory->get('drupal_purview.settings');
    $baseUrl = rtrim($config->get('rest_endpoint'), '/');
    $url = "{$baseUrl}/datamap/api/atlas/v2/types/typedefs";

    try {
      $response = $this->httpClient->get($url, [
        'headers' => [
          'Authorization' => "Bearer {$accessToken}",
          'Accept' => 'application/json',
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      if (!isset($data['businessMetadataDefs']) || !is_array($data['businessMetadataDefs'])) {
        return NULL;
      }

      $definitions = [];

      $enumLookup = [];
      foreach ($data['enumDefs'] ?? [] as $enumDef) {
        $name = $enumDef['name'] ?? NULL;
        if (!$name) {
          continue;
        }
        $enumLookup[$name] = array_values(array_map(
          static fn(array $el) => $el['value'] ?? NULL,
          $enumDef['elementDefs'] ?? []
        ));
        // Drop NULLs if any.
        $enumLookup[$name] = array_values(array_filter($enumLookup[$name], static fn($v) => $v !== NULL && $v !== ''));
      }

      foreach ($data['businessMetadataDefs'] as $group) {
        $group_name = $group['name'];
        $group_description = $group['description'] ?? '';
        foreach ($group['attributeDefs'] as $attr) {
          if (isset($attr['options']['isDisabled']) && $attr['options']['isDisabled'] === 'true') {
            continue;
          }

          $key = "{$group_name}.{$attr['name']}";

          // NEW: detect enum / array<enum> and attach options if available.
          $typeName = $attr['typeName'] ?? '';
          $isArray = FALSE;
          $baseType = $typeName;
          if (preg_match('/^array<([^>]+)>$/', $typeName, $m)) {
            $isArray = TRUE;
            $baseType = $m[1];
          }

          $definitions[$key] = [
            'group' => $group_name,
            'group_description' => $group_description,
            'field' => $attr['name'],
            'description' => $attr['description'] ?? '',
            'is_disabled' => $attr['options']['isDisabled'] ?? NULL,
          ];

          if ($baseType && isset($enumLookup[$baseType])) {
            $definitions[$key]['type'] = $isArray ? 'enum[]' : 'enum';
            $definitions[$key]['options'] = $enumLookup[$baseType];
          }
          else {
            $definitions[$key]['type'] = 'freeform';
          }
        }
      }

      if (!empty($definitions)) {
        $this->cache->set($cid, $definitions, strtotime('+1 day'), ['purview_managed_attribute_definitions']);
      }

      return $definitions;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get managed attribute definitions: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Builds a structured array of managed attributes grouped by category.
   *
   * @param array $managed_attributes
   *   The list of managed attributes from a glossary term.
   * @param array $attribute_definitions
   *   The definitions for all managed attributes.
   *
   * @return array
   *   A structured array grouped by attribute group, containing:
   *   - description: Group description.
   *   - fields: An array of fields with label, description, and values.
   */
  public function buildGroupedManagedAttributes(array $managed_attributes, array $attribute_definitions): array {
    $structured = [];

    foreach ($managed_attributes as $attribute) {
      $name = $attribute['name'];
      $value = !empty($attribute['value']) ? json_decode($attribute['value'], TRUE) : [];

      if (empty($value) || empty($attribute_definitions[$name])) {
        continue;
      }

      $definition = $attribute_definitions[$name];
      $group = $definition['group'];
      $group_description = $definition['group_description'];
      $field_label = $definition['field'];
      $field_description = $definition['description'];

      if (!isset($structured[$group])) {
        $structured[$group] = [
          'description' => $group_description,
          'fields' => [],
        ];
      }

      $structured[$group]['fields'][] = [
        'label' => $field_label,
        'description' => $field_description,
        'values' => $value,
      ];
    }

    return $structured;
  }

}
