<?php

declare(strict_types=1);

namespace Drupal\drupal_purview\Testing;

/**
 * Small helper for loading JSON fixtures from tests/fixtures.
 *
 * @internal Test-only utility.
 */
trait FixtureLoaderTrait {

  /**
   * Load a fixture file from the module.
   *
   * Looks in:
   * - <module>/tests/fixtures/<filename>
   * - <module>/fixtures/<filename>
   *
   * Works whether this trait lives in src/Testing or tests/src/TestHelpers.
   *
   * @throws \RuntimeException
   *   When the file is not found/readable.
   */
  protected function loadFixture(string $filename): string {
    $candidates = [];

    // If the kernel is up, use Drupal's extension service for a canonical path.
    if (defined('DRUPAL_ROOT') && \Drupal::hasContainer()) {
      $modulePath = \Drupal::service('extension.list.module')->getPath('drupal_purview');
      $candidates[] = DRUPAL_ROOT . '/' . $modulePath . '/tests/fixtures/' . $filename;
      $candidates[] = DRUPAL_ROOT . '/' . $modulePath . '/fixtures/' . $filename;
    }

    // Fallbacks based on this file's location.
    $moduleRoot = dirname(__DIR__, 3);
    $candidates[] = $moduleRoot . '/tests/fixtures/' . $filename;
    $candidates[] = $moduleRoot . '/fixtures/' . $filename;

    foreach ($candidates as $path) {
      if (is_readable($path)) {
        return file_get_contents($path);
      }
    }

    throw new \RuntimeException('Fixture not found. Tried: ' . implode(' | ', $candidates));
  }

  /**
   * Convenience helper for JSON fixtures.
   *
   * @return mixed
   *   Returns the JSON decoded file contents.
   */
  protected function loadJsonFixture(string $filename, bool $assoc = TRUE) {
    $json = $this->loadFixture($filename);
    return json_decode($json, $assoc);
  }

}
